/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.configuration.store;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Table;
import java.util.Map;
import org.appops.configuration.generator.ConfigWrapper;
import org.appops.marshaller.DescriptorType;

/**
 * In memory temporary configuration store which stores class configurations.
 *
 * @author deba
 * @version $Id: $Id
 */
public class ConfigurationStore {

  private static final String ENHANCER_BY_GUICE = "$$EnhancerByGuice";
  private static final String DEFAULT = "default";

  private Table<String, String, ConfigWrapper> configTable = HashBasedTable.create();

  /**
   * Adds configuration for a class of specific service.
   *
   * @param className Name of class for which configuration is to be stored.
   * @param configuration Configuration.
   */
  public void addConfiguration(String className, String configuration) {
    if (className.contains(ENHANCER_BY_GUICE)) {
      className = className.substring(0, className.indexOf(ENHANCER_BY_GUICE));
    }
    configTable.put(DEFAULT, className, new ConfigWrapper(configuration));
  }

  /**
   * Adds configuration for a class of specific service.
   *
   * @param className Name of class for which configuration is to be stored.
   * @param configuration Configuration.
   * @param descriptorType Type of configuration value. e.g. JSON,XML,YAML etc.
   */
  public void addConfiguration(String className, String configuration,
      DescriptorType descriptorType) {
    if (className.contains(ENHANCER_BY_GUICE)) {
      className = className.substring(0, className.indexOf(ENHANCER_BY_GUICE));
    }
    configTable.put(DEFAULT, className, new ConfigWrapper(configuration, descriptorType));
  }


  /**
   * Adds configuration for a class of specific service.
   *
   * @param serviceName Name of the service which class configuration is to be stored.
   * @param className Name of class for which configuration is to be stored.
   * @param configuration Configuration.
   */
  public void addConfiguration(String serviceName, String className, String configuration) {
    if (className.contains(ENHANCER_BY_GUICE)) {
      className = className.substring(0, className.indexOf(ENHANCER_BY_GUICE));
    }
    if (serviceName == null) {
      serviceName = DEFAULT;
    }
    configTable.put(serviceName, className, new ConfigWrapper(configuration));
  }


  /**
   * Fetches configuration for given class name.
   *
   * @param className Name of class of which configuration is to be fetched.
   * @return Configuration.
   */
  public String getConfiguration(String className) {
    if (className.contains(ENHANCER_BY_GUICE)) {
      className = className.substring(0, className.indexOf(ENHANCER_BY_GUICE));
    }
    Map<String, ConfigWrapper> classConfigMap = configTable.column(className);
    for (Map.Entry<String, ConfigWrapper> entry : classConfigMap.entrySet()) {
      if (entry.getValue() != null) {
        return entry.getValue().getConfig();
      }
    }
    return null;
  }

  /**
   * Fetches configuration for given class name.
   *
   * @param serviceName Name of the service which class configuration is to be stored.
   * @param className Name of class of which configuration is to be fetched.
   * @return Configuration.
   */
  public String getConfiguration(String serviceName, String className) {
    if (configTable.row(serviceName) != null
        && configTable.row(serviceName).get(className) != null) {
      return configTable.row(serviceName).get(className).getConfig();
    }
    return null;
  }

  /**
   * Fetches configuration for given class name.
   *
   * @param className Name of class of which configuration is to be fetched.
   * @return Configuration.
   */
  public ConfigWrapper getConfigWrapper(String className) {
    if (className.contains(ENHANCER_BY_GUICE)) {
      className = className.substring(0, className.indexOf(ENHANCER_BY_GUICE));
    }
    Map<String, ConfigWrapper> classConfigMap = configTable.column(className);
    for (Map.Entry<String, ConfigWrapper> entry : classConfigMap.entrySet()) {
      if (entry.getValue() != null) {
        return entry.getValue();
      }
    }
    return null;
  }



  /**
   * Fetches all configurations for given service name.
   *
   * @param serviceName Name of service which configurations is to be fetched.
   * @return Configurations map.
   */
  public Map<String, ConfigWrapper> getAllServiceConfig(String serviceName) {
    return configTable.row(serviceName);
  }



  /**
   * Checks if configuration for provided class is present in store or not.
   *
   * @param className Name of class for which configuration availability is to be checked.
   * @return Boolean result, true if configuration is present, false otherwise.
   */
  public boolean isConfigurationPresent(String className) {
    return configTable.containsColumn(className);
  }

}
