/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.configuration.generator;

import com.google.inject.Inject;
import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import org.appops.configuration.exception.ConfigurationException;
import org.appops.core.annotation.Config;
import org.appops.marshaller.DescriptorType;
import org.appops.marshaller.Marshaller;
import org.reflections.Reflections;
import org.reflections.scanners.FieldAnnotationsScanner;
import org.reflections.scanners.MethodParameterScanner;

/**
 * Generator class which scans all configuration classes of a project/service through annotation
 * scanning and generates single configuration of a project/service.
 *
 * @author deba
 * @version $Id: $Id
 */
public class ConfigJsonGenerator {

  private Marshaller marshaller;

  /**
   * Scans for classes that needs configurations and generates project configuration configuration
   * from information provided in annotations.
   *
   * @return Configuration created for the service.
   */
  public String generateConfiguration() {
    try {
      return generateConfiguration(null);

    } catch (Exception e) {
      throw new ConfigurationException(e);
    }
  }

  /**
   * Scans for classes that needs configurations and generates project configuration configuration
   * from information provided in annotations.
   *
   * @param descriptorType {@link org.appops.marshaller.DescriptorType} to marshall object
   * @return Configuration created for the service.
   */
  public String generateConfiguration(DescriptorType descriptorType) {
    if (descriptorType == null) {
      descriptorType = DescriptorType.JSON;
    }
    Map<String, Object> configMap = new HashMap<>();
    try {


      for (Class<?> clazz : classesTobeConfigured()) {
        configMap.put(clazz.getCanonicalName(), clazz.newInstance());
      }
      return getMarshaller().marshall(configMap, descriptorType);
    } catch (Exception e) {
      throw new ConfigurationException(e);
    }
  }

  public Set<Class<?>> findAnnotatedClasses(Class<? extends Annotation> annotation) {
    Reflections reflections =
        new Reflections("", new FieldAnnotationsScanner(), new MethodParameterScanner());
    Set<Class<?>> configClasses = reflections.getTypesAnnotatedWith(annotation);
    return configClasses;
  }



  /**
   * Scans for {@link org.appops.core.annotation.Config} annotation on classes.
   *
   * @return Annotated classes.
   */
  public Set<Class<?>> classesTobeConfigured() {
    Reflections reflections =
        new Reflections("", new FieldAnnotationsScanner(), new MethodParameterScanner());
    Set<Class<?>> annotatedClasses = new HashSet<>();
    findFromConstructors(reflections, annotatedClasses);
    findFromMethods(reflections, annotatedClasses);
    return annotatedClasses;
  }

  /**
   * Finds annotated configuration classes from method parameters.
   *
   * @param reflections Java reflections to fetch method metadata.
   * @param annotatedClasses Master configuration class set.
   * @return Updated master set.
   */
  private Set<Class<?>> findFromMethods(Reflections reflections, Set<Class<?>> annotatedClasses) {
    Set<Method> methods = reflections.getMethodsWithAnyParamAnnotated(Config.class);
    for (Method method : methods) {
      scanParameters(annotatedClasses, method.getParameters());
    }
    return annotatedClasses;
  }

  /**
   * Finds annotated configuration classes from constructor parameters.
   *
   * @param reflections Java reflections to fetch constructor metadata.
   * @param annotatedClasses Master configuration class set.
   * @return Updated master set.
   */
  private Set<Class<?>> findFromConstructors(Reflections reflections,
      Set<Class<?>> annotatedClasses) {
    Set<Constructor> annoatatedConstructors =
        reflections.getConstructorsWithAnyParamAnnotated(Config.class);

    for (Constructor constructor : annoatatedConstructors) {
      scanParameters(annotatedClasses, constructor.getParameters());
    }
    return annotatedClasses;
  }

  /**
   * Scans parameter information to fetch configurations.
   *
   * @param annotatedClasses Master configuration class set.
   * @param Array of parameters which is to be scanned.
   * @return Updated master set.
   */
  private Set<Class<?>> scanParameters(Set<Class<?>> annotatedClasses, Parameter[] parameters) {

    for (Parameter parameter : parameters) {
      if (parameter.isAnnotationPresent(Config.class)) {
        annotatedClasses.add(parameter.getType());
      }
    }
    return annotatedClasses;
  }

  /**
   * <p>
   * Getter for the field <code>marshaller</code>.
   * </p>
   *
   * @return a {@link org.appops.marshaller.Marshaller} object.
   */
  public Marshaller getMarshaller() {
    return marshaller;
  }

  /**
   * <p>
   * Setter for the field <code>marshaller</code>.
   * </p>
   *
   * @param marshaller a {@link org.appops.marshaller.Marshaller} object.
   */
  @Inject
  public void setMarshaller(Marshaller marshaller) {
    this.marshaller = marshaller;
  }


}
