/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.marshaller.json;

import com.fasterxml.jackson.annotation.JsonAutoDetect.Visibility;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import javax.xml.bind.DatatypeConverter;
import org.appops.core.response.Blob;
import org.appops.marshaller.exception.MarshalException;
import org.appops.marshaller.exception.UnmarshalException;
import org.json.JSONObject;

/**
 * <p>
 * JacksonJsonizer class.
 * </p>
 *
 * @author deba
 * @version $Id: $Id
 */
public class JacksonJsonizer implements Jsonizer {

  private ObjectMapper jsonMapper;

  /**
   * <p>
   * Constructor for JacksonJsonizer.
   * </p>
   */
  public JacksonJsonizer() {
    jsonMapper = createJsonMapper();
  }

  /** {@inheritDoc} */
  @Override
  public String toJson(Object object) {
    try {
       if (object.getClass().getCanonicalName().equals(Blob.class.getCanonicalName())) {
       jsonMapper.setVisibility(PropertyAccessor.FIELD, Visibility.ANY);
       }
      return jsonMapper.writeValueAsString(object).trim();
    } catch (JsonProcessingException e) {
      throw new MarshalException(e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public <T> T fromJson(String json, Class<T> clazz) {
    try {
      if (clazz.getCanonicalName() != Blob.class.getCanonicalName()) {
        return jsonMapper.readValue(json, clazz);
      }
      return (T) createCustomBlob(json);
    } catch (JsonProcessingException e) {
      throw new UnmarshalException(e);
    }
  }

  private Blob createCustomBlob(String json) {
    Blob blob = new Blob();
    JSONObject jsonObj = new JSONObject(json);
     
    
    String base64String =jsonObj.has("inputStream")? ((JSONObject) jsonObj.get("inputStream")).get("buf").toString():null;
    if(base64String !=null) {
      byte[] bytes = DatatypeConverter.parseBase64Binary(base64String);
      InputStream inputStream = new ByteArrayInputStream(bytes);
      blob.setInputStream(inputStream);
    }
    blob.setMimeType((String) jsonObj.get("mimeType"));
    blob.setNameOfFile((String) jsonObj.get("nameOfFile"));
   
    return blob;
  }

  /**
   * Creates and configures ObjectMapper.
   *
   * @return ObjectMapper instance.
   */
  private ObjectMapper createJsonMapper() {
    ObjectMapper mapper = new ObjectMapper();
    mapper.setVisibility(PropertyAccessor.FIELD, Visibility.ANY);
    mapper.enable(SerializationFeature.INDENT_OUTPUT);
    // mapper.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS,false);
    mapper.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
    mapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);

    mapper.enable(DeserializationFeature.ACCEPT_EMPTY_STRING_AS_NULL_OBJECT);
    mapper.setSerializationInclusion(Include.NON_EMPTY);
    // SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd_HH:mm:ss");
    // formatter.setTimeZone(TimeZone.getTimeZone("Asia/Kolkata"));
    // mapper.setDateFormat(formatter);
    // mapper.activateDefaultTypingAsProperty(new AppopsPolymorphicTypeValidator(),
    // DefaultTyping.NON_CONCRETE_AND_ARRAYS, "$T");
    return mapper;

  }

}
