/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.configuration;

import com.google.inject.Module;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.appops.core.deployment.DeploymentMode;
import org.appops.core.exception.CoreException;

/**
 * Service modules configuration.
 *
 * @author deba
 * @version $Id: $Id
 */
public class ModuleConfig {
  private Set<Class<? extends Module>> slimModules;
  private Set<Class<? extends Module>> implModules;


  /**
   * <p>
   * Constructor for ModuleConfig.
   * </p>
   */
  public ModuleConfig() {
    slimModules = new HashSet<>();
    implModules = new HashSet<>();
  }

  /**
   * <p>
   * Constructor for ModuleConfig.
   * </p>
   *
   * @param moduleConfig a {@link org.appops.configuration.ModuleConfig} object.
   */
  public ModuleConfig(ModuleConfig moduleConfig) {
    this();
    merge(moduleConfig);
  }



  /**
   * <p>
   * Getter for the field <code>slimModules</code>.
   * </p>
   *
   * @return a {@link java.util.Set} object.
   */
  public Set<Class<? extends Module>> getSlimModules() {
    return slimModules;
  }

  /**
   * Merges module configuration with this(current) configuration.
   *
   * @param config configuration to be merge.
   * @return {@link org.appops.configuration.ModuleConfig} with all modules.
   */
  public ModuleConfig merge(ModuleConfig config) {
    if (config.getSlimModules() != null) {
      slimModules.addAll(config.getSlimModules());
    }
    implModules.addAll(config.getImplModules());
    return this;
  }

  /**
   * <p>
   * Setter for the field <code>slimModules</code>.
   * </p>
   *
   * @param slimModules a {@link java.util.Set} object.
   */
  public void setSlimModules(Set<Class<? extends Module>> slimModules) {
    this.slimModules = slimModules;
  }

  /**
   * <p>
   * Getter for the field <code>implModules</code>.
   * </p>
   *
   * @return a {@link java.util.Set} object.
   */
  public Set<Class<? extends Module>> getImplModules() {
    return implModules;
  }

  /**
   * <p>
   * Setter for the field <code>implModules</code>.
   * </p>
   *
   * @param implModules a {@link java.util.Set} object.
   */
  public void setImplModules(Set<Class<? extends Module>> implModules) {
    this.implModules = implModules;
  }

  /**
   * Instantiates and provides list of modules enabled depending upon deployment mode passed.
   *
   * @param mode Deployment mode. (CLUBBED or STANDALONE)
   * @param classLoader Custom class loader.
   * @return {@link java.util.List} of {@link com.google.inject.Module}.
   */
  public List<Module> enabledModules(DeploymentMode mode, ClassLoader classLoader) {
    Logger logger = Logger.getLogger(this.getClass().getCanonicalName());
    List<Module> modulesToReturn = new ArrayList<>();
    try {
      if (DeploymentMode.CLUBBED.equals(mode)) {
        modulesToReturn.addAll(instantiateModules(getImplModules(), classLoader));
      } else if (DeploymentMode.STANDALONE.equals(mode)) {
        modulesToReturn.addAll(instantiateModules(getSlimModules(), classLoader));
      } else {
        throw new CoreException("Invalid Deployment Mode :" + mode.name());
      }
    } catch (Exception e) {
      logger.log(Level.SEVERE, "Exception in :: getModules() " + e);
    }
    return modulesToReturn;
  }

  /**
   * <p>
   * enabledModules.
   * </p>
   *
   * @param mode a {@link org.appops.core.deployment.DeploymentMode} object.
   * @return a {@link java.util.List} object.
   */
  public List<Module> enabledModules(DeploymentMode mode) {
    return this.enabledModules(mode, null);
  }

  /**
   * Add create and add module instance to the list passed.
   *
   * @param moduleClassNames Set of module class names.
   * @param classLoader Custom class loader.
   * @return List of module instances.
   */
  public List<Module> instantiateModules(Set<Class<? extends Module>> moduleClassNames,
      ClassLoader classLoader) {
    try {
      List<Module> modulesToReturn = new ArrayList<>();
      for (Class<? extends Module> moduleClazz : moduleClassNames) {
        modulesToReturn.add((Module) moduleClazz.newInstance());
      }
      return modulesToReturn;
    } catch (InstantiationException | IllegalAccessException e) {
      throw new CoreException(e);
    }

  }

}
