/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.configuration.guice;

import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.Map;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.appops.configuration.generator.ConfigWrapper;
import org.appops.configuration.provider.ConfigurationProvider;
import org.appops.configuration.provider.RunningServiceContextProvider;
import org.appops.core.annotation.Config;
import org.appops.marshaller.Marshaller;
import com.google.inject.Inject;

/**
 * Method intercepter class which intercepts method in which configuration is expected to be
 * injected.
 *
 * @author deba
 * @version $Id: $Id
 */
public class ConfigProvisionInterceptor implements MethodInterceptor {

  private ConfigurationProvider configurationProvider;
  private Marshaller marshaller;
  private RunningServiceContextProvider runningServiceContextProvider;

  /**
   * {@inheritDoc}
   *
   * Overridden method invocation.
   */
  @Override
  public Object invoke(MethodInvocation invocation) throws Throwable {

    Method m = invocation.getMethod();
    Parameter[] parameters = m.getParameters();
    Object[] args = invocation.getArguments();
    Map<String, ConfigWrapper> config = null;

    for (int i = 0; i < parameters.length; i++) {
      Parameter parameter = parameters[i];
      if (parameter.isAnnotationPresent(Config.class)) {
        config = configurationProvider
            .getConfigWrapperCollection(parameter.getType().getCanonicalName());
        for (Map.Entry<String, ConfigWrapper> entry : config.entrySet()) {
          ConfigWrapper configWrapper=entry.getValue();
          if (configWrapper != null && configWrapper.getConfig() != null
              && !configWrapper.getConfig().isEmpty()) {
            Object paramValue = getMarshaller().unmarshall(configWrapper.getConfig(),
                parameter.getType(), configWrapper.getDescriptorType());
            args[i] = paramValue;
          }
        }
      }
    }
    return invocation.proceed();
  }

  /**
   * <p>
   * setCustomProvisionListener.
   * </p>
   *
   * @param configurationProvider a {@link org.appops.configuration.provider.ConfigurationProvider}
   *        object.
   */
  @Inject
  public void setCustomProvisionListener(ConfigurationProvider configurationProvider) {
    this.configurationProvider = configurationProvider;
  }

  /**
   * <p>
   * Getter for the field <code>marshaller</code>.
   * </p>
   *
   * @return a {@link org.appops.marshaller.Marshaller} object.
   */
  public Marshaller getMarshaller() {
    return marshaller;
  }

  /**
   * <p>
   * Setter for the field <code>marshaller</code>.
   * </p>
   *
   * @param marshaller a {@link org.appops.marshaller.Marshaller} object.
   */
  @Inject
  public void setMarshaller(Marshaller marshaller) {
    this.marshaller = marshaller;
  }

  public RunningServiceContextProvider getRunningServiceContextProvider() {
    return runningServiceContextProvider;
  }

  @Inject
  public void setRunningServiceContextProvider(
      RunningServiceContextProvider runningServiceContextProvider) {
    this.runningServiceContextProvider = runningServiceContextProvider;
  }
}
