/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.configuration.loader;

import com.google.inject.Inject;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import org.apache.commons.io.FileUtils;
import org.appops.configuration.exception.ConfigurationException;
import org.appops.configuration.slimimpl.SlimImplStructure;
import org.appops.configuration.store.ConfigurationStore;
import org.appops.core.deployment.ConfigMap;
import org.appops.marshaller.DescriptorType;
import org.appops.marshaller.Marshaller;

/**
 * Loader class which load service configurations from a source(e.g. File).
 *
 * @author deba
 * @version $Id: $Id
 */
public class ConfigurationLoader {
  private ConfigurationStore configurationStore;
  private Marshaller marshaller;

  /**
   * <p>
   * Constructor for ConfigurationLoader.
   * </p>
   *
   * @param configurationStore a {@link org.appops.configuration.store.ConfigurationStore} object.
   * @param marshaller a {@link org.appops.marshaller.Marshaller} object.
   */
  @Inject
  public ConfigurationLoader(ConfigurationStore configurationStore, Marshaller marshaller) {
    setConfigurationStore(configurationStore);
    setMarshaller(marshaller);
  }

  /**
   * Load service configurations from file.
   *
   * @param file Service configuration file.
   */
  public void loadConfigurationsFromFile(File file) {
    loadConfigurationsFromFile(null, file, false);
  }

  /**
   * Load service configurations from file and save into store.
   *
   * @param serviceName Name of the service which class configuration is to be stored.
   * @param file Service configuration file.
   */
  public void loadConfigurationsFromFile(String serviceName, File file) {
    loadConfigurationsFromFile(serviceName, file, true);
  }

  /**
   * Load service configurations from file and save into store.
   * 
   * @param serviceName Name of the service which class configuration is to be stored.
   * @param file Service configuration file.
   * @param isServiceNameExpected is service name is expected or not.
   */
  private void loadConfigurationsFromFile(String serviceName, File file,
      boolean isServiceNameExpected) {
    if (isServiceNameExpected && (serviceName == null || serviceName.isEmpty())) {
      throw new ConfigurationException(
          "Use another method(a method without service name parameter) to load configuration.");
    }
    DescriptorType descriptorType = DescriptorType.fromExtension(file.getName());
    try {
      String marshalledData = FileUtils.readFileToString(file, StandardCharsets.UTF_8);
      SlimImplStructure slimImpConfig =
          getMarshaller().unmarshall(marshalledData, SlimImplStructure.class, descriptorType);
      ConfigMap configMap = slimImpConfig.getConfig();
      if (configMap != null && !configMap.isEmpty()) {
        for (String key : configMap.keySet()) {
          if (key != null) {
            String configJson = getMarshaller().marshall(configMap.get(key), DescriptorType.YML);

            getConfigurationStore().addConfiguration(serviceName, key, configJson);
          }
        }
      }
    } catch (IOException e) {
      throw new ConfigurationException(e);
    }
  }

  /**
   * <p>
   * Getter for the field <code>configurationStore</code>.
   * </p>
   *
   * @return a {@link org.appops.configuration.store.ConfigurationStore} object.
   */
  public ConfigurationStore getConfigurationStore() {
    return configurationStore;
  }

  private void setConfigurationStore(ConfigurationStore configurationStore) {
    this.configurationStore = configurationStore;
  }

  /**
   * <p>
   * Getter for the field <code>marshaller</code>.
   * </p>
   *
   * @return a {@link org.appops.marshaller.Marshaller} object.
   */
  public Marshaller getMarshaller() {
    return marshaller;
  }

  /**
   * <p>
   * Setter for the field <code>marshaller</code>.
   * </p>
   *
   * @param marshaller a {@link org.appops.marshaller.Marshaller} object.
   */
  @Inject
  public void setMarshaller(Marshaller marshaller) {
    this.marshaller = marshaller;
  }
}
