/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.configuration.slimimpl;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Table;
import com.google.inject.Module;
import java.util.Set;
import org.appops.core.constant.ConfigType;

public class SlimImplConfigTable {
  private Table<String, ConfigType, SlimImplStructure> configTable = HashBasedTable.create();

  /**
   * Fetches configuration for given service name.
   * 
   * @param serviceName Name of the service which class configuration is to be fetch.
   * @param type configuration object type {@link ConfigType}
   * @return a {@link SlimImplStructure} object.
   */
  public SlimImplStructure getConfigurationTable(String serviceName, ConfigType type) {
    if (configTable.row(serviceName) != null && configTable.row(serviceName).get(type) != null) {
      return configTable.row(serviceName).get(type);
    }
    return null;
  }

  /**
   * @return the configTable
   */
  public Table<String, ConfigType, SlimImplStructure> getConfigTable() {
    return configTable;
  }

  /**
   * Add module configurations of the service.
   * 
   * @param serviceName Name of the service which module configuration is to be stored.
   * @param type type of {@link ConfigType}.
   * @param modules list of modules to be add.
   */
  public void addModuleConfig(String serviceName, ConfigType type,
      Set<Class<? extends Module>> modules) {
    if (ConfigType.SLIM == type) {
      addSlimModules(serviceName, modules);
    } else if (ConfigType.IMPL == type) {
      addImplModules(serviceName, modules);
    }
  }

  /**
   * Add slim module configurations of the service.
   * 
   * @param serviceName Name of the service which module configuration is to be stored.
   * @param modules list of modules to be add.
   */
  public void addSlimModules(String serviceName, Set<Class<? extends Module>> modules) {
    if (getConfigurationTable(serviceName, ConfigType.SLIM) != null) {
      getConfigurationTable(serviceName, ConfigType.SLIM).setModules(modules);
    } else {
      SlimImplStructure structure = new SlimImplStructure();
      structure.setModules(modules);
      configTable.put(serviceName, ConfigType.SLIM, structure);
    }
  }

  /**
   * Add impl module configurations of the service.
   * 
   * @param serviceName Name of the service which module configuration is to be stored.
   * @param modules list of modules to be add.
   */
  public void addImplModules(String serviceName, Set<Class<? extends Module>> modules) {
    if (getConfigurationTable(serviceName, ConfigType.IMPL) != null) {
      getConfigurationTable(serviceName, ConfigType.IMPL).setModules(modules);
    } else {
      SlimImplStructure structure = new SlimImplStructure();
      structure.setModules(modules);
      configTable.put(serviceName, ConfigType.IMPL, structure);
    }
  }

  /**
   * Adds configuration for a class of specific service.
   * 
   * @param serviceName Name of the service which module configuration is to be stored.
   * @param configType configuration object type {@link ConfigType}
   * @param canonicalName canonical name of class for which configuration is to be stored.
   * @param configuration Configuration json of class
   */
  public void addConfiguration(String serviceName, ConfigType configType, String canonicalName,
      String configuration) {
    if (getConfigurationTable(serviceName, configType) != null) {
      getConfigurationTable(serviceName, configType).getConfig().put(canonicalName, configuration);
    } else {
      SlimImplStructure structure = new SlimImplStructure();
      structure.getConfig().put(canonicalName, configuration);
      configTable.put(serviceName, configType, structure);
    }
  }

}
