/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.core.constant;

import java.io.Serializable;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.TreeSet;

/**
 * Base class for iterable code constants defined under appops. To define a
 * custom application constant, the custom Constant class should be created and
 * derived from this class.
 *
 * @author deba
 * @version $Id: $Id
 */
public abstract class IterableConstant<T extends Serializable>
		implements java.io.Serializable, Comparable<IterableConstant<T>>, Constant {

	/**
	 * Constant value
	 */
	private T value;
	/**
	 * 
	 */
	private static final HashMap<String, TreeSet> types = new HashMap<>();

	/**
	 * <p>
	 * Constructor for IterableConstant.
	 * </p>
	 *
	 * @param value a T object.
	 */
	protected IterableConstant(T value) {
		this.value = value;
		storeType(this);
	}

	/** {@inheritDoc} */
	@Override
	public int compareTo(IterableConstant<T> o) {
		T thisValue = value();
		T val = o.value();

		if (thisValue instanceof Comparable && val instanceof Comparable) {
			return ((Comparable) thisValue).compareTo(val);
		}
		return 0;
	}

	/**
	 * Provides actual value stored in constant.
	 *
	 * @return Returns constant value.
	 */
	public T value() {
		return value;
	}

	/**
	 * Instance of constant will be registered to enumeration and other features.
	 *
	 * @param constant A constant instance/value that is to be registered.
	 */
	private void storeType(IterableConstant constant) {
		String className = constant.getClass().getName();

		TreeSet<IterableConstant> values;

		synchronized (types) { // avoid race condition for creating inner table
			values = types.computeIfAbsent(className, k -> new TreeSet<>());
		}

		values.add(constant);
	}

	/** {@inheritDoc} */
	@Override
	public boolean equals(Object obj) {
		if (obj instanceof IterableConstant) {
			if (this.value() != null) {
				return (this.value().equals(((IterableConstant) obj).value()));
			} else if (((IterableConstant) obj).value() == null) {
				return true;
			}
		}
		return false;
	}

	/** {@inheritDoc} */
	@Override
	public int hashCode() {
		return super.hashCode();
	}

	/**
	 * Creates an enumeration of constant values.
	 *
	 * @return An enumeration of constant values.
	 */
	public Enumeration elements() {
		String className = getClass().getName();
		TreeSet values = types.get(className);

		if (values != null) {
			return Collections.enumeration(values);
		} else {
			return Collections.emptyEnumeration();
		}
	}

	/**
	 * <p>
	 * Getter for the field <code>value</code>.
	 * </p>
	 *
	 * @return a T object.
	 */
	public T getValue() {
		return value;
	}

	/**
	 * <p>
	 * Setter for the field <code>value</code>.
	 * </p>
	 *
	 * @param value a T object.
	 */
	public void setValue(T value) {
		this.value = value;
	}
}
