/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.core.service.signature;


import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.appops.core.service.annotation.ServiceOp;
import org.appops.core.service.meta.InterfaceMeta;
import org.appops.core.service.meta.ServiceOpMeta;


/**
 * <p>ServiceOpSignatureBuilder class.</p>
 *
 * @author deba
 * @version $Id: $Id
 */
public class ServiceOpSignatureBuilder {



  /**
   * Constructs and provides signature of method passed.
   *
   * @param c containing interface.
   * @param m method in the interface.
   * @return signature string.
   */
  public String buildMethodSignature(Class<?> c, Method m) {
    String signature = "_" + c.getCanonicalName();
    String name = m.getName();
    if (m.isAnnotationPresent(ServiceOp.class)) {
      String friendly = m.getAnnotation(ServiceOp.class).friendly();
      if (friendly != null && !friendly.isEmpty()) {
        name = friendly;
      }
    }
    signature += "_" + name;
    signature += getParamSignatures(m.getParameterTypes());
    return signature;
  }

  /**
   * Constructs and provides signature of op meta passed.
   *
   * @param opMeta Service operation meta instance.
   * @return signature string.
   */
  public String buildOpSignature(ServiceOpMeta opMeta) {
    String signature = "";
    InterfaceMeta interfaceMeta = opMeta.getParent();
    if (interfaceMeta != null) {
      String interfaceName = (interfaceMeta.getQualifiedClassName() != null
          && !interfaceMeta.getQualifiedClassName().isEmpty())
              ? interfaceMeta.getQualifiedClassName()
              : interfaceMeta.getName();
      signature += ("_" + interfaceName);
    }
    String name = opMeta.getName();
    String friendly = opMeta.getFriendly();
    if (friendly != null && !friendly.isEmpty()) {
      name = friendly;
    }
    signature += "_" + name;
    signature += getParamSignatures(opMeta.parameterTypes());
    return signature;
  }

  /**
   * Uses parameter Simple Type Names i.e. String instead of java.lang.String for brevity.
   *
   * @param paramTypes Array of method parameter types.
   * @return Returns method parameter signature.
   */
  private String getParamSignatures(Class<?>[] paramTypes) {
    List<String> typeList = new ArrayList<>();
    for (Class<?> type : paramTypes) {
      typeList.add(type.getCanonicalName());
    }
    return getParamSignatures(typeList);

  }

  /**
   * Uses parameter Simple Type Names i.e. String instead of java.lang.String for brevity.
   *
   * @param paramTypes Collection of method parameter types.
   * @return Returns method parameter signature.
   */
  private String getParamSignatures(Collection<String> parameterTypes) {
    String paramSig = "";

    for (String typeName : parameterTypes) {
      paramSig += "_" + typeName;
    }
    return paramSig;
  }

}
