/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.marshaller;

import com.google.inject.Inject;
import org.appops.marshaller.exception.MarshalException;
import org.appops.marshaller.exception.UnmarshalException;
import org.appops.marshaller.json.Jsonizer;
import org.appops.marshaller.xml.XmlMarshaller;
import org.appops.marshaller.yaml.YamlMarshaller;

/**
 * Implementation of {@link org.appops.marshaller.Marshaller}.
 *
 * @author deba
 * @version $Id: $Id
 */
public class MarshallerImpl implements Marshaller {

  private Jsonizer jsonizer;
  private XmlMarshaller xmlMarshaller;
  private YamlMarshaller yamlMarshaller;


  /** {@inheritDoc} */
  @Override
  public String marshall(Object object, DescriptorType marshaller) {
    if (marshaller == null) {
      throw new MarshalException("Marshaller type is null.");
    }

    if (DescriptorType.JSON.equals(marshaller)) {
      return getJsonizer().toJson(object);
    } else if (DescriptorType.XML.equals(marshaller)) {
      return getXmlMarshaller().toXml(object);
    } else if (DescriptorType.YML.equals(marshaller)) {
      return getYamlMarshaller().toYaml(object);
    } else {
      throw new MarshalException("Marshaller type " + marshaller.value()
          + " Not supported yet. Supported types - JSON, XML, YML");
    }
  }

  /** {@inheritDoc} */
  @Override
  public <T> T unmarshall(String marshalledData, Class<T> clazz, DescriptorType unmarshaller) {
    if (unmarshaller == null) {
      throw new UnmarshalException("Marshaller type is null.");
    }
    if (DescriptorType.JSON.equals(unmarshaller)) {
      return getJsonizer().fromJson(marshalledData, clazz);
    } else if (DescriptorType.XML.equals(unmarshaller)) {
      return getXmlMarshaller().fromXml(marshalledData, clazz);
    } else if (DescriptorType.YML.equals(unmarshaller)) {
      return getYamlMarshaller().fromYaml(marshalledData, clazz);
    } else {
      throw new UnmarshalException("Marshaller type " + unmarshaller.value()
          + " Not supported yet. Supported types - JSON, XML, YML");
    }
  }

  /**
   * <p>Getter for the field <code>jsonizer</code>.</p>
   *
   * @return a {@link org.appops.marshaller.json.Jsonizer} object.
   */
  public Jsonizer getJsonizer() {
    return jsonizer;
  }

  /**
   * <p>Setter for the field <code>jsonizer</code>.</p>
   *
   * @param jsonizer a {@link org.appops.marshaller.json.Jsonizer} object.
   */
  @Inject
  public void setJsonizer(Jsonizer jsonizer) {
    this.jsonizer = jsonizer;
  }

  /**
   * <p>Getter for the field <code>xmlMarshaller</code>.</p>
   *
   * @return a {@link org.appops.marshaller.xml.XmlMarshaller} object.
   */
  public XmlMarshaller getXmlMarshaller() {
    return xmlMarshaller;
  }

  /**
   * <p>Setter for the field <code>xmlMarshaller</code>.</p>
   *
   * @param xmlMarshaller a {@link org.appops.marshaller.xml.XmlMarshaller} object.
   */
  @Inject
  public void setXmlMarshaller(XmlMarshaller xmlMarshaller) {
    this.xmlMarshaller = xmlMarshaller;
  }

  /**
   * <p>Getter for the field <code>yamlMarshaller</code>.</p>
   *
   * @return a {@link org.appops.marshaller.yaml.YamlMarshaller} object.
   */
  public YamlMarshaller getYamlMarshaller() {
    return yamlMarshaller;
  }

  /**
   * <p>Setter for the field <code>yamlMarshaller</code>.</p>
   *
   * @param yamlMarshaller a {@link org.appops.marshaller.yaml.YamlMarshaller} object.
   */
  @Inject
  public void setYamlMarshaller(YamlMarshaller yamlMarshaller) {
    this.yamlMarshaller = yamlMarshaller;
  }

}
