/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.configuration.provider;

import com.google.inject.Inject;
import java.util.Map;
import org.appops.configuration.generator.ConfigWrapper;
import org.appops.configuration.store.ConfigurationStore;

/**
 * Provider class which provides class configuration with help of store.
 *
 * @author deba
 * @version $Id: $Id
 */
public class ConfigurationProvider {

	private ConfigurationStore configurationStore;

	/**
	 * <p>
	 * Constructor for ConfigurationProvider.
	 * </p>
	 *
	 * @param configurationStore a
	 *                           {@link org.appops.configuration.store.ConfigurationStore}
	 *                           object.
	 */
	@Inject
	public ConfigurationProvider(ConfigurationStore configurationStore) {
		this.configurationStore = configurationStore;
	}

	/**
	 * Fetches configuration for given class name.
	 *
	 * @param className Name of class of which configuration is to be fetched.
	 * @return Configuration json.
	 */
	public String getConfiguration(String className) {
		return configurationStore.getConfiguration(className);
	}

	/**
	 * Fetches configuration for given class name.
	 *
	 * @param serviceName Name of the service which class configuration is to be
	 *                    stored.
	 * @param className   Name of class of which configuration is to be fetched.
	 * @return Configuration json.
	 */
	public String getConfiguration(String serviceName, String className) {
		return configurationStore.getConfiguration(serviceName, className);
	}

	/**
	 * Fetches configuration for given class name.
	 *
	 * @param className Name of class of which configuration is to be fetched.
	 * @return Configuration .
	 */
	public ConfigWrapper getConfigWrapper(String className) {
		return configurationStore.getConfigWrapper(className);
	}
	   /**
     * Fetches configurations for given class name.
     *
     * @param className Name of class of which configuration is to be fetched.
     * @return Configurations .
     */
    public Map<String, ConfigWrapper> getConfigWrapperCollection(String className) {
        return configurationStore.getConfigWrapperCollection(className);
    }

	/**
	 * Fetches all configurations for given service name.
	 *
	 * @param serviceName Name of service which configurations is to be fetched.
	 * @return Configurations map.
	 */
	public Map<String, ConfigWrapper> getAllServiceConfiguration(String serviceName) {
		return configurationStore.getAllServiceConfig(serviceName);
	}

	/**
	 * Checks if configuration for provided class is present in store or not.
	 *
	 * @param className Name of class for which configuration availability is to be
	 *                  checked.
	 * @return Boolean result, true if configuration is present, false otherwise.
	 */
	public boolean isConfigurationPresent(String className) {
		return configurationStore.isConfigurationPresent(className);
	}

	/**
	 * Fetches configuration for given class name and serviceName.
	 *
	 * @param className Name of class of which configuration is to be fetched.
	 * @param serviceName current service name .
	 * @return Configuration .
	 */
	public ConfigWrapper getConfigWrapperByService(String canonicalName, String serviceName) {
		return configurationStore.getConfigWrapperByService(canonicalName, serviceName);
	}

}
