/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.logging.destination;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.ConsoleHandler;
import java.util.logging.FileHandler;
import java.util.logging.Filter;
import java.util.logging.Handler;
import java.util.logging.LogManager;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import org.apache.logging.log4j.LoggingException;
import org.appops.logging.logger.config.HandlerConfig;
import org.appops.logging.logger.config.LogFilter;
import org.appops.logging.logger.config.constant.FilterProperty;
import org.appops.logging.logger.config.constant.FilterReply;
import org.appops.logging.logger.config.constant.FilterType;
import org.appops.logging.logger.config.constant.HandlerAttribute;
import org.appops.logging.logger.config.constant.LoggerType;
import org.appops.logging.logger.formatter.JavaUtilLogFormatter;
import org.appops.logging.meta.Level;

/**
 * <p>JavaUtilLogger class.</p>
 *
 * @author deba
 * @version $Id: $Id
 */
public class JavaUtilLogger extends DestinationLogger<Logger> {
  private Logger internalLogger;


  /**
   * <p>Constructor for JavaUtilLogger.</p>
   */
  public JavaUtilLogger() {

  }

  /** {@inheritDoc} */
  @Override
  public LoggerType type() {
    return LoggerType.JUL;
  }

  /** {@inheritDoc} */
  @Override
  public boolean log(Level level, String message) {
    if (level == null) {
      level = org.appops.logging.meta.Level.ALL;
    }
    internalLogger().log(convertLevel(level.name()), message);
    return true;
  }

  /** {@inheritDoc} */
  @Override
  protected Logger internalLogger() {
    if (internalLogger == null) {
      internalLogger = LogManager.getLogManager().getLogger("");
    }
    return internalLogger;
  }

  /**
   * {@inheritDoc}
   *
   * Remove all handlers.
   */
  @Override
  public void removeAllHandlers() {
    for (Handler deafaultHandler : internalLogger().getHandlers()) {
      internalLogger().removeHandler(deafaultHandler);
    }
  }


  /**
   * {@inheritDoc}
   *
   * Override a {@link JavaUtilLogger} .
   */
  @Override
  public void addHandlers(List<HandlerConfig> handlers) {
    if (handlers == null || handlers.isEmpty()) {
      throw new LoggingException("Please provide valid log handlers.");
    }
    for (HandlerConfig handler : handlers) {
      addHandler(handler);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void addHandler(HandlerConfig handler) {
    if (handler == null) {
      throw new LoggingException("Log handler cannot be null.");
    }
    if (handler.enabled()) {
      switch (handler.name()) {
        case CONSOLE:
          addConsoleHandler(handler);
          break;
        case FILE:
          addFileAppender(handler);
          break;
        default:
          throw new LoggingException("Please provide valid log handler.");
      }
    }
  }



  /**
   * Set configuration to console handler if configuration is empty or null it will use default
   * Configuration and use it.
   *
   * @param logHandler configuration object which is to set handler.
   */
  public void addConsoleHandler(HandlerConfig logHandler) {
    Map<HandlerAttribute, Object> config = logHandler.getConfig();

    ConsoleHandler handler = new ConsoleHandler();
    java.util.logging.Level level = config.get(HandlerAttribute.LEVEL) != null
        ? convertLevel(config.get(HandlerAttribute.LEVEL).toString())
        : java.util.logging.Level.ALL;
    handler.setLevel(level);

    JavaUtilLogFormatter julFormatter = new JavaUtilLogFormatter();
    if (config.get(HandlerAttribute.PATTERN) != null) {
      julFormatter.setFormat(formatPattern(config.get(HandlerAttribute.PATTERN).toString()));
    }
    handler.setFormatter(julFormatter);
    addFilters(handler, logHandler.getFilters());
    internalLogger().setLevel(level);
    internalLogger().addHandler(handler);
  }

  /**
   * Add file handler to logger and set configurations like log file name,max size.
   * 
   * @param handlerConfig configuration object which is to set handler.
   * 
   */
  private void addFileAppender(HandlerConfig handlerConfig) {
    Map<HandlerAttribute, Object> config = handlerConfig.getConfig();

    String filename = config.get(HandlerAttribute.FILENAME) != null
        ? config.get(HandlerAttribute.FILENAME).toString()
        : "JUL.log";
    Integer maxBackupIndex = config.get(HandlerAttribute.MAX_BACKUP_INDEX) != null
        ? Integer.parseInt(config.get(HandlerAttribute.MAX_BACKUP_INDEX).toString())
        : 2;
    java.util.logging.Level level = config.get(HandlerAttribute.LEVEL) != null
        ? convertLevel(config.get(HandlerAttribute.LEVEL).toString())
        : java.util.logging.Level.ALL;
    JavaUtilLogFormatter julFormatter = new JavaUtilLogFormatter();
    if (config.get(HandlerAttribute.PATTERN) != null) {
      julFormatter.setFormat(formatPattern(config.get(HandlerAttribute.PATTERN).toString()));

    }
    int fileSize = getFileSize(config.get(HandlerAttribute.MAXSIZE));

    try {
      FileHandler fileHandler = new FileHandler(filename, fileSize, maxBackupIndex, true);

      fileHandler.setLevel(level);
      fileHandler.setFormatter(julFormatter);
      addFilters(fileHandler, handlerConfig.getFilters());
      internalLogger().addHandler(fileHandler);
    } catch (Exception e) {
      System.out.println("Exception occurs while adding file handler::" + e);

    }
  }



  /**
   * Convert file size from KB/MB/GB into bytes.
   * 
   * @param object size which is to be convert
   * @return byte representation of provided file size.
   */
  private int getFileSize(Object object) {
    int defaultSize = 10 * 1024 * 1024;
    try {
      if (object != null) {
        String sizeString = String.valueOf(object);
        String unit = sizeString.substring(sizeString.length() - 2, sizeString.length());
        int size = Integer.parseInt(sizeString.substring(0, sizeString.length() - 2));

        switch (unit.toUpperCase()) {

          case "KB":
            return size * 1024;
          case "MB":
            return size * 1024 * 1024;
          case "GB":
            if (size < 2) {
              return size * 1024 * 1024 * 1024;
            } else {
              return defaultSize;
            }
          default:
            return defaultSize;

        }
      }
    } catch (Exception e) {
      System.out
          .println("Exception occured while converting file size it will use default size." + e);
    }
    return defaultSize;
  }

  /**
   * Convert {@link org.appops.logging.meta.Level} into {@link java.util.logging.Level} .
   * 
   * @param level log level
   * @return java.util.logging.Level
   */
  private java.util.logging.Level convertLevel(String level) {
    if (Level.WARN.name().equals(level) || Level.WARNING.name().equals(level)) {
      return java.util.logging.Level.WARNING;
    } else if (Level.SEVERE.name().equals(level)) {
      return java.util.logging.Level.SEVERE;
    } else if (Level.ALL.name().equals(level)) {
      return java.util.logging.Level.ALL;
    } else if (Level.INFO.name().equals(level)) {
      return java.util.logging.Level.INFO;
    } else if (Level.FINE.name().equals(level)) {
      return java.util.logging.Level.FINE;
    } else if (Level.OFF.name().equals(level)) {
      return java.util.logging.Level.OFF;
    } else {
      return java.util.logging.Level.INFO;
    }
  }

  /**
   * {@inheritDoc}
   *
   * It formats the given pattern into Java util logger format and return it.
   */
  @Override
  public String formatPattern(String pattern) {
    pattern = pattern.trim().replaceAll(" +", " ");
    if (pattern.contains("%d")) {
      if (!pattern.contains("%d{")) {
        pattern = pattern.replaceAll("%d", "%1\\$tF %1\\$tT");
      } else {
        pattern = pattern.replaceFirst("%d", "");
        pattern = pattern.replaceAll("\\{", "");
        pattern = pattern.replaceAll("\\}", "");

      }
    }
    pattern = pattern.replaceAll("%yyyy", "%1\\$tY");
    pattern = pattern.replaceAll("%MMMM", "%1\\$tB");
    pattern = pattern.replaceAll("%MMM", "%1\\$tb");
    pattern = pattern.replaceAll("%MM", "%1\\$tm");
    pattern = pattern.replaceAll("%dd", "%1\\$td");

    pattern = pattern.replaceAll("%HH", "%1\\$tk");
    pattern = pattern.replaceAll("%hh", "%1\\$tl");
    pattern = pattern.replaceAll("%mm", "%1\\$tM");
    pattern = pattern.replaceAll("%ss", "%1\\$tS");
    pattern = pattern.replaceAll("%logger", "%2\\$s");
    pattern = pattern.replaceAll("%level", "%4\\$s");
    pattern = pattern.replaceAll("%msg", "%5\\$s");
    return pattern;
  }

  /**
   * It filters out all log messages with provided filter.
   * 
   * @param handler handler to add filter on.
   * @param filters list of filter which want to apply.
   */
  private void addFilters(Handler handler, ArrayList<LogFilter> filters) {
    try {
      for (LogFilter filterConfig : filters) {
        if (FilterType.RegEx.equals(filterConfig.getName())) {
          handler.setFilter(addLogFilter(filterConfig));
        }
      }
    } catch (Exception e) {
      throw new LoggingException(
          "Exception Occured while adding filter to ::" + getClass().getCanonicalName() + e);
    }
  }

  /**
   * Each Handler can have a filter associated with it. The Handler will call the decide method to
   * check if a given LogRecord should be published. If decide returns 0, the LogRecord will be
   * discarded.
   * 
   * @param config configuration object to be set.
   * @return instance of Filter.
   */
  private Filter addLogFilter(LogFilter config) {
    return new Filter() {

      @Override
      public boolean isLoggable(LogRecord record) {
        if (config != null) {
          if (FilterProperty.LOGGER_NAME.equals(config.getApplyOn())
              && record.getLoggerName().matches(config.getExpression())) {
            return getvalue(config.getOnMatch());
          } else if (FilterProperty.MESSAGE.equals(config.getApplyOn())
              && record.getMessage().toString().matches(config.getExpression())) {
            return getvalue(config.getOnMatch());
          }
        }
        return !getvalue(config.getOnMatch());
      }
    };
  }

  /**
   * This return the possible replies that a filtering component can return.
   * 
   * @param onMatch value which want to convert into Boolean.
   * @return false if onMatch is DENY otherwise true.
   */
  private boolean getvalue(FilterReply onMatch) {
    switch (onMatch) {
      case ACCEPT:
        return true;
      case DENY:
        return false;
      default:
        return true;
    }
  }
}
