/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.logging.logger.store;

import java.util.Collection;
import org.appops.logging.meta.Level;
import org.appops.logging.meta.LogMeta;

/**
 * <p>LogStore interface.</p>
 *
 * @author deba
 * @version $Id: $Id
 */
public interface LogStore {

  /**
   * Adds log record to store.
   *
   * @param logRecord Record to be added to store.
   */
  public void addLogRecord(LogMeta logRecord);

  /**
   * Fetches all log records.
   *
   * @return Collection of log meta records.
   */
  public Collection<LogMeta> getAllLogRecords();

  /**
   * Provides filtered log records.
   *
   * @param keyword Keyword or string to be looked for in log message.
   * @return Collection of filtered log records.
   */
  public Collection<LogMeta> filterByKeyword(String keyword);

  /**
   * Provides filtered log records.
   *
   * @param level Log level.
   * @return Collection of filtered log records.
   */
  public Collection<LogMeta> filterByLevel(Level level);

  /**
   * Fetches log records by page.
   *
   * @param startIndex Start index of page.
   * @param pageSize Number of records to be included in page.
   * @return Collection of log meta records by page.
   */
  public Collection<LogMeta> getLogRecordsByPage(Integer startIndex, Integer pageSize);

  /**
   * Provides filtered log records.
   *
   * @param startDate Start date of filter range.
   * @return Collection of filtered log records.
   */
  public Collection<LogMeta> getLogRecordsByStartDate(String startDate);

  /**
   * Provides filtered log records.
   *
   * @param startDate Start date of filter range.
   * @param endDate End date of filter range.
   * @return Collection of filtered log records.
   */
  public Collection<LogMeta> getLogRecords(String startDate, String endDate);

  /**
   * Provides filtered log records.
   *
   * @param startTime Start time of filter range.
   * @return Collection of filtered log records.
   */
  public Collection<LogMeta> getLogRecordsByStartTime(String startTime);

  /**
   * Provides filtered log records.
   *
   * @param startTime Start time of filter range.
   * @param endTime End time of filter range.
   * @return Collection of filtered log records.
   */
  public Collection<LogMeta> getLogRecordsByStartTimeEndTime(String startTime, String endTime);

  /**
   * Provides filtered log records.
   *
   * @param startDate Start date of filter range.
   * @param endDate End date of filter range.
   * @param startTime Start time of filter range.
   * @param endTime End time of filter range.
   * @return Collection of filtered log records.
   */
  public Collection<LogMeta> getLogRecordsByDateAndTime(String startDate, String endDate,
      String startTime, String endTime);

  /**
   * Removes older logs according to configuration provided from service configuration.
   */
  public void removeOlderLogs();
}
