/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.logging.destination;

import com.google.inject.Inject;
import com.google.inject.Provider;
import java.util.HashMap;
import org.appops.core.annotation.Config;
import org.appops.logging.logger.config.LoggerConfig;
import org.appops.logging.logger.config.constant.LoggerType;

/**
 * Storage where all {@link org.appops.logging.logger.config.constant.LoggerType} is stored in map in {@link org.appops.logging.logger.config.constant.LoggerType} as key and its
 * provider as value.
 *
 * @author deba
 * @version $Id: $Id
 */
public class DestinationLoggerStore {
  private HashMap<LoggerType, DestinationLogger<?>> destinationLoggers;
  private LoggerConfig config;
  private Provider<NoOpLogger> noOpLogger;

  /**
   * <p>Constructor for DestinationLoggerStore.</p>
   */
  public DestinationLoggerStore() {

  }

  /**
   * <p>register.</p>
   *
   * @param logger a {@link org.appops.logging.destination.DestinationLogger} object.
   */
  public void register(DestinationLogger<?> logger) {
    getDestinationLoggers().put(logger.type(), logger);
  }

  /**
   * Injected provider of logger and register into destinationLogger map.
   *
   * @param log4jLogger provider of {@link org.appops.logging.destination.Log4jLogger}
   * @param javaUtilLogger provider {@link org.appops.logging.destination.JavaUtilLogger}
   * @param logbackLogger provider of{@link org.appops.logging.destination.LogbackLogger}
   */
  @Inject
  public DestinationLoggerStore(Provider<Log4jLogger> log4jLogger,
      Provider<JavaUtilLogger> javaUtilLogger, Provider<LogbackLogger> logbackLogger) {
    register(log4jLogger.get());
    register(javaUtilLogger.get());
    register(logbackLogger.get());
  }

  /**
   * Fetch {@link org.appops.logging.destination.DestinationLogger} from {@link org.appops.logging.logger.config.LoggerConfig} and return instance.
   *
   * @return instance of specific {@link org.appops.logging.destination.DestinationLogger}.
   */
  public DestinationLogger get() {
    LoggerType destinationLogger = getConfig().getDestinationLogger();
    if (destinationLogger != null) {
      return getDestinationLoggers().get(destinationLogger);
    }
    return getNoOpLogger();
  }

  /**
   * To fetch specific {@link org.appops.logging.destination.DestinationLogger} instance.
   *
   * @param logger logger name which to be fetch
   * @return instance of specific {@link org.appops.logging.destination.DestinationLogger}
   */
  public DestinationLogger<?> get(LoggerType logger) {
    if (logger != null) {
      return getDestinationLoggers().get(logger);
    }
    return getNoOpLogger();
  }

  /**
   * It returns {@link java.util.HashMap} which contains {@link org.appops.logging.logger.config.constant.LoggerType} as key and
   * {@link org.appops.logging.destination.DestinationLogger} as a value.
   *
   * @return {@link java.util.HashMap} which contains {@link org.appops.logging.logger.config.constant.LoggerType} as key and {@link org.appops.logging.destination.DestinationLogger}
   *         as a value
   */
  public HashMap<LoggerType, DestinationLogger<?>> getDestinationLoggers() {
    if (destinationLoggers == null) {
      destinationLoggers = new HashMap<LoggerType, DestinationLogger<?>>();
    }
    return destinationLoggers;
  }

  /**
   * <p>Setter for the field <code>destinationLoggers</code>.</p>
   *
   * @param destinationLoggers a {@link java.util.HashMap} object.
   */
  public void setDestinationLoggers(HashMap<LoggerType, DestinationLogger<?>> destinationLoggers) {
    this.destinationLoggers = destinationLoggers;
  }

  /**
   * <p>Getter for the field <code>config</code>.</p>
   *
   * @return a {@link org.appops.logging.logger.config.LoggerConfig} object.
   */
  public LoggerConfig getConfig() {
    return config;
  }

  /**
   * <p>Setter for the field <code>config</code>.</p>
   *
   * @param config a {@link org.appops.logging.logger.config.LoggerConfig} object.
   */
  @Inject
  public void setConfig(@Config LoggerConfig config) {
    this.config = config;
  }

  /**
   * <p>Getter for the field <code>noOpLogger</code>.</p>
   *
   * @return a {@link org.appops.logging.destination.NoOpLogger} object.
   */
  public NoOpLogger getNoOpLogger() {
    return noOpLogger.get();
  }

  /**
   * <p>Setter for the field <code>noOpLogger</code>.</p>
   *
   * @param noOpLogger a {@link com.google.inject.Provider} object.
   */
  @Inject
  public void setNoOpLogger(Provider<NoOpLogger> noOpLogger) {
    this.noOpLogger = noOpLogger;
  }

}
