/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.logging.destination;

import ch.qos.logback.classic.Logger;
import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.encoder.PatternLayoutEncoder;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.Appender;
import ch.qos.logback.core.ConsoleAppender;
import ch.qos.logback.core.filter.Filter;
import ch.qos.logback.core.rolling.RollingFileAppender;
import ch.qos.logback.core.rolling.SizeBasedTriggeringPolicy;
import ch.qos.logback.core.rolling.TimeBasedRollingPolicy;
import ch.qos.logback.core.spi.FilterReply;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.apache.logging.log4j.LoggingException;
import org.appops.logging.logger.config.HandlerConfig;
import org.appops.logging.logger.config.LogFilter;
import org.appops.logging.logger.config.constant.FilterProperty;
import org.appops.logging.logger.config.constant.FilterType;
import org.appops.logging.logger.config.constant.HandlerAttribute;
import org.appops.logging.logger.config.constant.LoggerType;
import org.appops.logging.meta.Level;
import org.slf4j.LoggerFactory;



/**
 * <p>LogbackLogger class.</p>
 *
 * @author deba
 * @version $Id: $Id
 */
public class LogbackLogger extends DestinationLogger<Logger> {
  private LoggerContext loggerContext;
  private Logger internalLogger;

  /**
   * <p>Constructor for LogbackLogger.</p>
   */
  public LogbackLogger() {

  }

  /** {@inheritDoc} */
  @Override
  public boolean log(Level level, String message) {
    ch.qos.logback.classic.Level actuallevel;
    if (level == null) {
      actuallevel = convertLevel(null);
    } else {
      actuallevel = convertLevel(level.name());
    }
    internalLogger().log(null, null, actuallevel.toInt() / 1000, message, null, null);
    return true;
  }

  /** {@inheritDoc} */
  @Override
  public LoggerType type() {
    return LoggerType.LOGBACK;
  }

  /** {@inheritDoc} */
  @Override
  public void removeAllHandlers() {
    internalLogger().detachAndStopAllAppenders();
  }

  /**
   * {@inheritDoc}
   *
   * Override a {@link LogbackLogger} .
   */
  @Override
  public void addHandlers(List<HandlerConfig> handlers) {
    if (handlers == null || handlers.isEmpty()) {
      throw new LoggingException("Please provide valid log handlers.");
    }
    for (HandlerConfig handler : handlers) {
      addHandler(handler);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void addHandler(HandlerConfig handler) {
    if (handler == null) {
      throw new LoggingException("Please provide valid log handlers.");
    }
    if (handler.enabled()) {
      switch (handler.name()) {
        case CONSOLE:
          addConsoleAppender(handler);
          break;
        case FILE:
          addFileAppender(handler);
          break;
        default:
          throw new LoggingException("Please provide valid log handler.");
      }
    }

  }

  /**
   * Return {@link LoggerContext}.
   * 
   * @return logger context;
   */
  private LoggerContext getLoggerContext() {
    if (loggerContext == null) {
      loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
    }
    return loggerContext;
  }

  /** {@inheritDoc} */
  @Override
  protected Logger internalLogger() {
    if (internalLogger == null) {
      internalLogger = getLoggerContext().getLogger(Logger.ROOT_LOGGER_NAME);
    }
    return internalLogger;
  }


  /**
   * Set configuration to console handler if configuration is empty or null it will use default
   * Configuration and use it.
   * 
   * @param handlerConfig handlerConfig configuration object which is to set handler
   */
  private void addConsoleAppender(HandlerConfig handlerConfig) {
    Map<HandlerAttribute, Object> config = handlerConfig.getConfig();

    PatternLayoutEncoder ple = new PatternLayoutEncoder();
    String pattern = "%date %level [%thread] %logger{10} [%file:%line] %msg%n";
    if (config.get(HandlerAttribute.PATTERN) != null) {
      pattern = formatPattern(config.get(HandlerAttribute.PATTERN).toString());
    }

    ple.setPattern(pattern);
    ple.setContext(getLoggerContext());
    ple.start();

    ConsoleAppender<ILoggingEvent> consoleAppender = new ConsoleAppender<ILoggingEvent>();
    consoleAppender.setContext(getLoggerContext());
    consoleAppender.setName(handlerConfig.name().name());
    consoleAppender.setEncoder(ple);
    addFilters(consoleAppender, handlerConfig.getFilters());
    consoleAppender.start();
    ch.qos.logback.classic.Level level = config.get(HandlerAttribute.LEVEL) != null
        ? convertLevel(config.get(HandlerAttribute.LEVEL).toString())
        : ch.qos.logback.classic.Level.ALL;

    internalLogger().setLevel(level);
    internalLogger().addAppender(consoleAppender);
  }

  /**
   * Add file handler to logger and set configurations like log file name,max size.
   * 
   * @param logHandler configuration object which is to set handler.
   * 
   */
  private void addFileAppender(HandlerConfig logHandler) {
    Map<HandlerAttribute, Object> config = logHandler.getConfig();

    String filename = config.get(HandlerAttribute.FILENAME) != null
        ? config.get(HandlerAttribute.FILENAME).toString()
        : "logback.log";

    RollingFileAppender<ILoggingEvent> logFileAppender = new RollingFileAppender<ILoggingEvent>();
    logFileAppender.setContext(getLoggerContext());
    logFileAppender.setName(logHandler.name().name());
    logFileAppender.setAppend(true);
    logFileAppender.setFile(filename);

    PatternLayoutEncoder patternEncoder = getPatternEncoder(config.get(HandlerAttribute.PATTERN));
    TimeBasedRollingPolicy<ILoggingEvent> rollingPolicy = getTimeBasedPolicy(logFileAppender,
        filename, config.get(HandlerAttribute.MAX_BACKUP_INDEX));

    SizeBasedTriggeringPolicy<ILoggingEvent> sizeBasedTriggeringPolicy =
        getSizeBasedPolicy(config.get(HandlerAttribute.MAXSIZE));

    logFileAppender.setEncoder(patternEncoder);
    logFileAppender.setRollingPolicy(rollingPolicy);
    logFileAppender.setTriggeringPolicy(sizeBasedTriggeringPolicy);
    addFilters(logFileAppender, logHandler.getFilters());

    logFileAppender.start();

    ch.qos.logback.classic.Level logbackLevel = config.get(HandlerAttribute.LEVEL) != null
        ? convertLevel(config.get(HandlerAttribute.LEVEL).toString())
        : ch.qos.logback.classic.Level.ALL;

    internalLogger().setLevel(logbackLevel);
    internalLogger().addAppender(logFileAppender);
  }

  /**
   * Populate {@link SizeBasedTriggeringPolicy} instance fot file handler.
   * 
   * @param maxSize max file size to set.
   * 
   * @return populated instance of SizeBasedTriggeringPolicy.
   */
  private SizeBasedTriggeringPolicy<ILoggingEvent> getSizeBasedPolicy(Object maxSize) {
    SizeBasedTriggeringPolicy<ILoggingEvent> sizeBasedPolicy =
        new SizeBasedTriggeringPolicy<ILoggingEvent>();
    sizeBasedPolicy.setContext(getLoggerContext());
    if (maxSize != null) {
      sizeBasedPolicy.setMaxFileSize(maxSize.toString());
    }
    sizeBasedPolicy.start();
    return sizeBasedPolicy;

  }

  /**
   * Populate {@link TimeBasedRollingPolicy} instance fot file handler.
   * 
   * @param logFileAppender parent i.e.log file appender.
   * @param filename log file name.
   * @param maxBackupIndex Number of max indexed log file
   * 
   * @return populated instance of TimeBasedRollingPolicy.
   */
  private TimeBasedRollingPolicy<ILoggingEvent> getTimeBasedPolicy(
      RollingFileAppender<ILoggingEvent> logFileAppender, String filename, Object maxBackupIndex) {
    TimeBasedRollingPolicy<ILoggingEvent> timeBasedPolicy =
        new TimeBasedRollingPolicy<ILoggingEvent>();
    timeBasedPolicy.setFileNamePattern("%d." + filename);
    timeBasedPolicy.setMaxHistory(3);
    timeBasedPolicy.setParent(logFileAppender);
    timeBasedPolicy.setContext(getLoggerContext());
    timeBasedPolicy.start();
    return timeBasedPolicy;
  }

  /**
   * Create {@link PatternLayoutEncoder} object.
   * 
   * @param object pattern string
   * @return encoder of handler.
   */
  private PatternLayoutEncoder getPatternEncoder(Object pattern) {
    String defaultPattern = "%date %level [%thread] %logger{10} [%file:%line] %msg%n";
    if (pattern != null) {
      defaultPattern = formatPattern(pattern.toString());
    }
    PatternLayoutEncoder ple = new PatternLayoutEncoder();
    ple.setPattern(defaultPattern);
    ple.setContext(getLoggerContext());
    ple.start();
    return ple;

  }

  /**
   * Convert {@link org.appops.logging.meta.Level} into {@link ch.qos.logback.classic.Level} .
   * 
   * @param level log level
   * @return ch.qos.logback.classic.Level
   */
  private ch.qos.logback.classic.Level convertLevel(String level) {
    if (Level.WARN.name().equals(level) || Level.WARNING.name().equals(level)) {
      return ch.qos.logback.classic.Level.WARN;
    } else if (Level.ERROR.name().equals(level)) {
      return ch.qos.logback.classic.Level.ERROR;
    } else if (Level.ALL.name().equals(level)) {
      return ch.qos.logback.classic.Level.TRACE;
    } else if (Level.INFO.name().equals(level)) {
      return ch.qos.logback.classic.Level.INFO;
    } else if (Level.OFF.name().equals(level)) {
      return ch.qos.logback.classic.Level.OFF;
    } else {
      return ch.qos.logback.classic.Level.INFO;
    }
  }

  /**
   * {@inheritDoc}
   *
   * It formats the given pattern into Logback logger format and return it.
   */
  @Override
  protected String formatPattern(String pattern) {
    pattern = pattern.trim().replaceAll(" +", " ");
    if (pattern.contains("%d")) {
      if (!pattern.contains("%d{")) {
        pattern = pattern.replaceAll("%d", "%d{yyyy-MM-dd HH:mm:ss}");
      }
    }
    pattern = pattern.replaceAll("%yyyy", "yyyy");
    pattern = pattern.replaceAll("%MMMM", "MMMM");
    pattern = pattern.replaceAll("%MMM", "MMM");
    pattern = pattern.replaceAll("%MM", "MM");
    pattern = pattern.replaceAll("%dd", "dd");

    pattern = pattern.replaceAll("%HH", "HH");
    pattern = pattern.replaceAll("%hh", "hh");
    pattern = pattern.replaceAll("%mm", "mm");
    pattern = pattern.replaceAll("%ss", "ss");
    return pattern;
  }

  /**
   * It filters out all log messages with provided filter.
   * 
   * @param appender appender to add filter on.
   * @param filters list of filter which want to apply.
   */
  private void addFilters(Appender<ILoggingEvent> appender, ArrayList<LogFilter> filters) {
    try {
      for (LogFilter filterConfig : filters) {
        if (FilterType.RegEx.equals(filterConfig.getName())) {
          appender.addFilter(addLogFilter(filterConfig));
        }
      }
    } catch (Exception e) {
      throw new LoggingException(
          "Exception Occured while adding filter to ::" + getClass().getCanonicalName() + e);
    }
  }

  /**
   * Each Handler can have a filter associated with it. The Handler will call the decide method to
   * check if a given LogRecord should be published. If decide returns 0, the LogRecord will be
   * discarded.
   * 
   * @param config configuration object to be set.
   * @return {@link Filter}.
   */
  private Filter<ILoggingEvent> addLogFilter(LogFilter config) {
    return new Filter<ILoggingEvent>() {
      @Override
      public FilterReply decide(ILoggingEvent event) {
        if (config != null) {
          if (FilterProperty.LOGGER_NAME.equals(config.getApplyOn())
              && event.getLoggerName().matches(config.getExpression())) {
            return onMatch(config.getOnMatch());
          } else if (FilterProperty.MESSAGE.equals(config.getApplyOn())
              && event.getMessage().toString().matches(config.getExpression())) {
            return onMatch(config.getOnMatch());
          }
        }
        return onMismatch(config.getOnMatch());
      }
    };
  }


  /**
   * This return the possible replies that a filtering component in logback can return.
   * 
   * @param onMatch value which want to convert into FilterReply.
   * @return specific {@link FilterReply}
   */
  private FilterReply onMatch(org.appops.logging.logger.config.constant.FilterReply onMatch) {
    return convertFilterReply(onMatch, false);
  }

  /**
   * It convert onMatch value into onMismatch and return the possible replies that a filtering
   * component in logback can return on mismatch.Ex if onMatch is {@link FilterReply.DENY} convert
   * into {@link FilterReply.ACCEPT} and vice versa.
   * 
   * @param onMatch value which want to convert into FilterReply.
   * @return mismatch value
   */
  private FilterReply onMismatch(org.appops.logging.logger.config.constant.FilterReply onMatch) {
    return convertFilterReply(onMatch, true);
  }

  /**
   * This return the possible replies that a filtering component in logback can return.
   * 
   * @param onMatch value which want to convert into FilterReply.
   * @param invert boolean value to identify is onMatch or onMismatch
   * @return specific {@link FilterReply}
   */
  private FilterReply convertFilterReply(
      org.appops.logging.logger.config.constant.FilterReply onMatch, boolean invert) {
    switch (onMatch) {
      case ACCEPT:
        return invert ? FilterReply.DENY : FilterReply.ACCEPT;
      case DENY:
        return invert ? FilterReply.ACCEPT : FilterReply.DENY;
      default:
        return FilterReply.NEUTRAL;
    }
  }
}
