/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.logging.impl;

import com.google.inject.Inject;
import com.google.inject.Provider;
import org.appops.core.annotation.Config;
import org.appops.logging.destination.DestinationLogger;
import org.appops.logging.logger.Logger;
import org.appops.logging.logger.config.LoggerConfig;
import org.appops.logging.logger.store.LogStore;
import org.appops.logging.meta.Level;
import org.appops.logging.meta.LogMeta;


/**
 * Base implementation for all logger implementations supported in appops framework.
 *
 * @author deba
 * @version $Id: $Id
 */
public class LoggerImpl implements Logger {

  private LogMeta meta;
  private Provider<DestinationLogger> destinationLogger;
  private LoggerConfig config;
  private LogStore logStore;
  public String loggerClassName;


  /** {@inheritDoc} */
  @Override
  public Logger withLevel(Level level) {
    getMeta().addAttribute(LogMeta.LEVEL.value(), level);
    return this;
  }

  /**
   * Checks if log level provided is enabled or not.
   *
   * @param level Level to be checked.
   * @return Boolean result, true if level is enabled.
   */
  public boolean isEnabled(Level level) {
    Level enabledLevel = getConfig().getLevel();
    return !(Level.OFF.equals(enabledLevel));
  }

  /** {@inheritDoc} */
  @Override
  public Logger withMessage(String message) {
    getMeta().addAttribute(LogMeta.MESSAGE.value(), message);
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public Logger withMeta(LogMeta meta) {
    this.setMeta(meta);
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public Logger withMeta(String key, Object value) {
    getMeta().addAttribute(key, value);
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public boolean log() {
    Level level = (Level) getMeta().getAttributeValue(LogMeta.LEVEL.value());
    String message = (String) getMeta().getAttributeValue(LogMeta.MESSAGE.value());

    // save log into store
    getLogStore().addLogRecord(getMeta());
    setMeta(null);
    return getDestinationLogger().log(level, message);
  }

  /**
   * Provides log meta, creates new meta instance if it is null.
   *
   * @return Log meta.
   */
  public LogMeta getMeta() {
    if (meta == null) {
      meta = new LogMeta();
      if (loggerClassName != null) {
        meta.addAttribute(LogMeta.LOGGERNAME.value(), loggerClassName);
      }
    }
    return meta;
  }

  /**
   * <p>Setter for the field <code>meta</code>.</p>
   *
   * @param meta a {@link org.appops.logging.meta.LogMeta} object.
   */
  public void setMeta(LogMeta meta) {
    this.meta = meta;
  }

  /**
   * <p>Getter for the field <code>destinationLogger</code>.</p>
   *
   * @return a {@link org.appops.logging.destination.DestinationLogger} object.
   */
  public DestinationLogger getDestinationLogger() {
    return destinationLogger.get();
  }

  /**
   * <p>Setter for the field <code>destinationLogger</code>.</p>
   *
   * @param destinationLogger a {@link com.google.inject.Provider} object.
   */
  @Inject
  public void setDestinationLogger(Provider<DestinationLogger> destinationLogger) {
    this.destinationLogger = destinationLogger;
  }

  /**
   * <p>Getter for the field <code>config</code>.</p>
   *
   * @return a {@link org.appops.logging.logger.config.LoggerConfig} object.
   */
  public LoggerConfig getConfig() {
    return config;
  }

  /**
   * <p>Setter for the field <code>config</code>.</p>
   *
   * @param config a {@link org.appops.logging.logger.config.LoggerConfig} object.
   */
  @Inject
  public void setConfig(@Config LoggerConfig config) {
    this.config = config;
  }

  /**
   * <p>Getter for the field <code>logStore</code>.</p>
   *
   * @return a {@link org.appops.logging.logger.store.LogStore} object.
   */
  public LogStore getLogStore() {
    return logStore;
  }

  /**
   * <p>Setter for the field <code>logStore</code>.</p>
   *
   * @param logStore a {@link org.appops.logging.logger.store.LogStore} object.
   */
  @Inject
  public void setLogStore(LogStore logStore) {
    this.logStore = logStore;
  }

  /** {@inheritDoc} */
  @Override
  public Logger setLoggerName(String loggerName) {
    this.loggerClassName = loggerName;
    return this;
  }

}
