/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.logging.logger.store;

import com.google.inject.Inject;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TreeMap;
import org.appops.logging.meta.Level;
import org.appops.logging.meta.LogMeta;
import org.appops.marshaller.Marshaller;

/**
 * <p>LogStoreImpl class.</p>
 *
 * @author deba
 * @version $Id: $Id
 */
public class LogStoreImpl implements LogStore {

  private Marshaller marshaller;

  private Map<Integer, LogMeta> logRecords = new TreeMap<Integer, LogMeta>();

  /** {@inheritDoc} */
  @Override
  public void addLogRecord(LogMeta logRecord) {
    logRecords.put(logRecords.size() + 1, logRecord);

  }

  /**
   * {@inheritDoc}
   *
   * Filter log records by keyword.
   */
  @Override
  public Collection<LogMeta> filterByKeyword(String keyword) {
    List<LogMeta> listToReturn = new ArrayList<>();
    for (LogMeta logRecord : logRecords.values()) {
      Map<String, Object> logMetaMap = logRecord.getMetaMap();
      for (Object value : logMetaMap.values()) {
        if (String.valueOf(value).contains(keyword)) {
          listToReturn.add(logRecord);
        }
      }
    }
    return listToReturn;
  }

  /**
   * {@inheritDoc}
   *
   * Get log records by pagination.
   */
  @Override
  public Collection<LogMeta> getLogRecordsByPage(Integer startIndex, Integer pageSize) {
    List<LogMeta> listToReturn = new ArrayList<>();
    int endIndex = startIndex + pageSize;
    Iterator<Integer> iterator = logRecords.keySet().iterator();
    while (iterator.hasNext()) {
      Integer index = iterator.next();
      if (index >= startIndex && index < endIndex) {
        listToReturn.add(logRecords.get(index));
      }
    }
    return listToReturn;
  }

  /**
   * {@inheritDoc}
   *
   * Filter log records by level.
   */
  @Override
  public Collection<LogMeta> filterByLevel(Level expectedLevel) {
    List<LogMeta> listToReturn = new ArrayList<>();
    if (expectedLevel == null) {
      return listToReturn;
    }
    for (LogMeta logRecord : logRecords.values()) {
      Level level = (Level) logRecord.getMetaMap().get(LogMeta.LEVEL.value());
      if (expectedLevel.equals(level) || expectedLevel.equals(Level.ALL)) {
        listToReturn.add(logRecord);
      }
    }
    return listToReturn;
  }

  /**
   * {@inheritDoc}
   *
   * Get filtered log records by start date.
   */
  @Override
  public Collection<LogMeta> getLogRecordsByStartDate(String startDateOfLog) {

    List<LogMeta> listToReturn = new ArrayList<>();
    Date startDate = null;

    startDateOfLog = startDateOfLog.substring(0, 9);

    DateFormat format = new SimpleDateFormat("MMMddyyyy", Locale.ENGLISH);
    try {
      startDate = format.parse(startDateOfLog);
    } catch (ParseException e) {
      e.printStackTrace();
    }


    for (LogMeta logRecord : logRecords.values()) {
      Map<String, Object> logMetaMap = logRecord.getMetaMap();

      for (Object key : logMetaMap.keySet()) {
        if (key.equals("datetime")) {
          Date date2;

          String dateValue = (String) logMetaMap.get(key);
          dateValue = dateValue.substring(0, 13);
          dateValue = dateValue.replace(",", "");

          DateFormat format2 = new SimpleDateFormat("MMM dd yyyy", Locale.ENGLISH);
          try {
            date2 = format2.parse(dateValue);

            if (date2.compareTo(startDate) >= 0) {
              listToReturn.add(logRecord);
            }
          } catch (ParseException e) {
            e.printStackTrace();
          }
        }
      }
    }

    return listToReturn;
  }

  /**
   * {@inheritDoc}
   *
   * Get filtered log records by start date and end date.
   */
  @Override
  public Collection<LogMeta> getLogRecords(String startLogDate, String endLogDate) {

    List<LogMeta> listToReturn = new ArrayList<>();
    Date startDate = null;
    Date endDate = null;

    startLogDate = startLogDate.substring(0, 9);
    endLogDate = endLogDate.substring(0, 9);

    DateFormat format = new SimpleDateFormat("MMMddyyyy", Locale.ENGLISH);
    try {
      startDate = format.parse(startLogDate);
      endDate = format.parse(endLogDate);
    } catch (ParseException e) {
      e.printStackTrace();
    }

    for (LogMeta logRecord : logRecords.values()) {
      Map<String, Object> logMetaMap = logRecord.getMetaMap();

      for (Object key : logMetaMap.keySet()) {
        if (key.equals("datetime")) {
          Date logListDate;

          String dateValue = (String) logMetaMap.get(key);
          dateValue = dateValue.substring(0, 13);
          dateValue = dateValue.replace(",", "");

          DateFormat format2 = new SimpleDateFormat("MMM dd yyyy", Locale.ENGLISH);
          try {
            logListDate = format2.parse(dateValue);

            if (logListDate.compareTo(startDate) >= 0 && logListDate.compareTo(endDate) <= 0) {
              listToReturn.add(logRecord);
            }
          } catch (ParseException e) {
            e.printStackTrace();
          }
        }
      }
    }

    return listToReturn;
  }

  /**
   * {@inheritDoc}
   *
   * Get filtered log records by start time.
   */
  @Override
  public Collection<LogMeta> getLogRecordsByStartTime(String startTime) {

    List<LogMeta> listToReturn = new ArrayList<>();
    LocalTime startLocalTime = null;

    startLocalTime = LocalTime.parse(startTime);

    for (LogMeta logRecord : logRecords.values()) {
      Map<String, Object> logMetaMap = logRecord.getMetaMap();

      for (Object key : logMetaMap.keySet()) {
        if (key.equals("datetime")) {
          LocalTime logListTime;

          String timeValue = (String) logMetaMap.get(key);
          timeValue = timeValue.substring(13, 21);

          logListTime = LocalTime.parse(timeValue);

          if (logListTime.compareTo(startLocalTime) >= 0) {
            listToReturn.add(logRecord);
          }
        }
      }
    }

    return listToReturn;
  }

  /**
   * {@inheritDoc}
   *
   * Get filtered log records by start time and end time.
   */
  @Override
  public Collection<LogMeta> getLogRecordsByStartTimeEndTime(String startTime, String endTime) {

    List<LogMeta> listToReturn = new ArrayList<>();
    LocalTime startLocalTime = null;
    LocalTime endLocalTime = null;

    startLocalTime = LocalTime.parse(startTime);
    endLocalTime = LocalTime.parse(endTime);

    for (LogMeta logRecord : logRecords.values()) {
      Map<String, Object> logMetaMap = logRecord.getMetaMap();

      for (Object key : logMetaMap.keySet()) {
        if (key.equals("datetime")) {
          LocalTime logsTime;

          String timeValue = (String) logMetaMap.get(key);
          timeValue = timeValue.substring(13, 21);

          logsTime = LocalTime.parse(timeValue);

          if (logsTime.compareTo(startLocalTime) >= 0 && logsTime.compareTo(endLocalTime) <= 0) {
            listToReturn.add(logRecord);
          }
        }
      }
    }

    return listToReturn;
  }

  /**
   * {@inheritDoc}
   *
   * Get filtered log records by date and time. The method invokes the
   * getFilteredDateListOfDateTime() method for date filtration. After that invokes
   * getFinalFilteredListOfDateTime() for the time filtration from the result of filtered date list.
   */
  @Override
  public Collection<LogMeta> getLogRecordsByDateAndTime(String startDateOfLog, String endDateOfLog,
      String startTime, String endTime) {

    Date startDate = null;
    Date endDate = null;

    startDateOfLog = startDateOfLog.substring(0, 9);
    endDateOfLog = endDateOfLog.substring(0, 9);

    DateFormat format = new SimpleDateFormat("MMMddyyyy", Locale.ENGLISH);
    try {
      startDate = format.parse(startDateOfLog);
      endDate = format.parse(endDateOfLog);
    } catch (ParseException e) {
      e.printStackTrace();
    }

    LocalTime startLocalTime = null;
    startLocalTime = LocalTime.parse(startTime);
    LocalTime endLocalTime = null;
    endLocalTime = LocalTime.parse(endTime);

    List<LogMeta> sortedDates = this.getFilteredDateListOfDateTime(startDate, endDate);

    List<LogMeta> listToReturn = new ArrayList<>();
    listToReturn = this.getFinalFilteredListOfDateTime(startLocalTime, endLocalTime, sortedDates);

    return listToReturn;
  }

  /**
   * Filters logs by date for date-time log filtration.
   *
   * @param startDateLog : Starting Date of Log
   * @param endDateLog : Ending Date of Log
   * @return filtered date list of date-time.
   */
  public List<LogMeta> getFilteredDateListOfDateTime(Date startDateLog, Date endDateLog) {

    List<LogMeta> sortedDateList = new ArrayList<>();

    for (LogMeta logRecord : logRecords.values()) {
      Map<String, Object> logMetaMap = logRecord.getMetaMap();

      for (Object key : logMetaMap.keySet()) {
        if (key.equals("datetime")) {
          Date logsDate;

          String dateValue = (String) logMetaMap.get(key);
          dateValue = dateValue.substring(0, 13);
          dateValue = dateValue.replace(",", "");

          DateFormat format2 = new SimpleDateFormat("MMM dd yyyy", Locale.ENGLISH);
          try {
            logsDate = format2.parse(dateValue);

            if (logsDate.compareTo(startDateLog) >= 0 && logsDate.compareTo(endDateLog) <= 0) {
              sortedDateList.add(logRecord);
            }
          } catch (ParseException e) {
            e.printStackTrace();
          }
        }
      }
    }

    return sortedDateList;
  }

  /**
   * Filters logs by time, after date is filtered of date-time log filtration.
   *
   * @param startTime : Starting Time of Log
   * @param endTime : Ending Time of Log
   * @param sortedDateList : filtered date list for date-time filtration.
   * @return final filtered list for date-time filtration.
   */
  public List<LogMeta> getFinalFilteredListOfDateTime(LocalTime startTime, LocalTime endTime,
      Collection<LogMeta> sortedDateList) {

    List<LogMeta> listToReturn = new ArrayList<>();

    for (LogMeta element : sortedDateList) {
      Map<String, Object> logMetaMap = element.getMetaMap();

      for (Object key : logMetaMap.keySet()) {
        if (key.equals("datetime")) {
          LocalTime logListTime;

          String timeValue = (String) logMetaMap.get(key);
          timeValue = timeValue.substring(13, 21);

          logListTime = LocalTime.parse(timeValue);

          if ((logListTime.compareTo(startTime) >= 0) && (logListTime.compareTo(endTime) <= 0)) {
            listToReturn.add(element);
          }
        }
      }
    }

    return listToReturn;
  }

  /** {@inheritDoc} */
  @Override
  public Collection<LogMeta> getAllLogRecords() {
    return logRecords.values();
  }


  /**
   * <p>Getter for the field <code>marshaller</code>.</p>
   *
   * @return a {@link org.appops.marshaller.Marshaller} object.
   */
  public Marshaller getMarshaller() {
    return marshaller;
  }


  /**
   * <p>Setter for the field <code>marshaller</code>.</p>
   *
   * @param marshaller a {@link org.appops.marshaller.Marshaller} object.
   */
  @Inject
  public void setMarshaller(Marshaller marshaller) {
    this.marshaller = marshaller;
  }


  /** {@inheritDoc} */
  @Override
  public void removeOlderLogs() {
    // TODO Auto-generated method stub
  }

}
