/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.logging.destination;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.apache.log4j.Appender;
import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.RollingFileAppender;
import org.apache.log4j.spi.Filter;
import org.apache.log4j.spi.LoggingEvent;
import org.apache.logging.log4j.LoggingException;
import org.appops.logging.logger.config.HandlerConfig;
import org.appops.logging.logger.config.LogFilter;
import org.appops.logging.logger.config.constant.FilterProperty;
import org.appops.logging.logger.config.constant.FilterReply;
import org.appops.logging.logger.config.constant.FilterType;
import org.appops.logging.logger.config.constant.HandlerAttribute;
import org.appops.logging.logger.config.constant.LoggerType;
import org.appops.logging.meta.Level;

/**
 * Add and manage handlers of {@link java.util.logging.Logger}.
 *
 * @author deba
 * @version $Id: $Id
 */
public class Log4jLogger extends DestinationLogger<Logger> {
  private Logger internalLogger;

  /** {@inheritDoc} */
  @Override
  public boolean log(org.appops.logging.meta.Level level, String message) {
    if (level == null) {
      level = org.appops.logging.meta.Level.ALL;
    }
    internalLogger().log(convertLevel(level.name()), message);
    return true;
  }

  /** {@inheritDoc} */
  @Override
  public LoggerType type() {
    return LoggerType.LOG4J;
  }

  /** {@inheritDoc} */
  @Override
  protected Logger internalLogger() {
    if (internalLogger == null) {
      internalLogger = Logger.getRootLogger();
    }
    return internalLogger;
  }

  /**
   * {@inheritDoc}
   *
   * Set configuration to specific handler.
   */
  public void addHandlers(List<HandlerConfig> handlers) {
    try {
      if (handlers == null || handlers.isEmpty()) {
        throw new LoggingException("Please provide valid log handlers.");
      }
      for (HandlerConfig handler : handlers) {
        addHandler(handler);
      }
    } catch (Exception e) {
      throw new LoggingException(
          "Exception Occured while adding handler to ::" + getClass().getCanonicalName() + e);
    }
  }

  /** {@inheritDoc} */
  @Override
  public void addHandler(HandlerConfig handler) {
    if (handler == null) {
      throw new LoggingException("Handler cant't be null");
    }
    if (handler.enabled()) {
      switch (handler.name()) {
        case CONSOLE:
          addConsoleAppender(handler);
          break;
        case FILE:
          addFileAppender(handler);
          break;
        default:
          throw new LoggingException("Please provide valid log handler.");
      }
    }
  }

  /** {@inheritDoc} */
  @Override
  public void removeAllHandlers() {
    internalLogger().removeAllAppenders();
  }


  /**
   * Set configuration to console handler if configuration is empty or null it will use default
   * Configuration and use it.
   * 
   * @param handler configuration object which is to set handler.
   */
  private void addConsoleAppender(HandlerConfig handler) {
    ConsoleAppender console = new ConsoleAppender(); // create appender
    // configure the appender
    Map<HandlerAttribute, Object> config = handler.getConfig();
    String pattern = config.get(HandlerAttribute.PATTERN) != null
        ? formatPattern(config.get(HandlerAttribute.PATTERN).toString())
        : "[%-5d] %-5p - %m%n";

    console.setLayout(new PatternLayout(pattern));
    org.apache.log4j.Level level = org.apache.log4j.Level.ALL;
    if (config.get(HandlerAttribute.LEVEL) != null) {
      level = convertLevel(config.get(HandlerAttribute.LEVEL).toString());
    }
    console.setThreshold(level);
    addFilters(console, handler.getFilters());

    console.activateOptions();
    console.setName((handler.name()).name());
    // add appender to any Logger (here is root)
    internalLogger().addAppender(console);
  }

  /**
   * Add file handler to logger and set configurations like log file name,max size.
   * 
   * @param handlerConfig configuration object which is to set handler.
   * 
   */
  private void addFileAppender(HandlerConfig handlerConfig) {
    Logger logger = internalLogger();

    Map<HandlerAttribute, Object> config = handlerConfig.getConfig();

    String filename = config.get(HandlerAttribute.FILENAME) != null
        ? config.get(HandlerAttribute.FILENAME).toString()
        : "log4j.log";
    String pattern = config.get(HandlerAttribute.PATTERN) != null
        ? formatPattern(config.get(HandlerAttribute.PATTERN).toString())
        : "[%-5d] %-5p - %m%n";

    String maxSize = config.get(HandlerAttribute.MAXSIZE) != null
        ? config.get(HandlerAttribute.MAXSIZE).toString()
        : "10MB";
    Integer maxBackupIndex = config.get(HandlerAttribute.MAX_BACKUP_INDEX) != null
        ? Integer.parseInt(config.get(HandlerAttribute.MAX_BACKUP_INDEX).toString())
        : 2;
    org.apache.log4j.Level level = org.apache.log4j.Level.ALL;
    if (config.get(HandlerAttribute.LEVEL) != null) {
      level = convertLevel(config.get(HandlerAttribute.LEVEL).toString());
    }
    PatternLayout layout = new PatternLayout(pattern);
    try {
      RollingFileAppender fileHandler = new RollingFileAppender();
      fileHandler.setThreshold(level);
      fileHandler.setFile(filename);
      fileHandler.setLayout(layout);
      fileHandler.setMaxFileSize(maxSize);
      fileHandler.setMaxBackupIndex(maxBackupIndex);
      fileHandler.setName((handlerConfig.name()).name());
      addFilters(fileHandler, handlerConfig.getFilters());

      fileHandler.activateOptions();
      logger.addAppender(fileHandler);
    } catch (Exception e) {
      throw new LoggingException("Exception occurs while adding file handler::" + e);
    }
  }

  /**
   * Convert {@link org.appops.logging.meta.Level} into {@link org.apache.log4j.Level} .
   * 
   * @param level log level
   * @return org.apache.log4j.Level
   */
  private org.apache.log4j.Level convertLevel(String level) {
    if (Level.WARN.name().equals(level) || Level.WARNING.name().equals(level)) {
      return org.apache.log4j.Level.WARN;
    } else if (Level.ERROR.name().equals(level)) {
      return org.apache.log4j.Level.ERROR;
    } else if (Level.ALL.name().equals(level)) {
      return org.apache.log4j.Level.ALL;
    } else if (Level.INFO.name().equals(level)) {
      return org.apache.log4j.Level.INFO;
    } else if (Level.OFF.name().equals(level)) {
      return org.apache.log4j.Level.OFF;
    } else {
      return org.apache.log4j.Level.INFO;
    }

  }

  /**
   * {@inheritDoc}
   *
   * It formats the given pattern into Log4J logger format and return it.
   */
  @Override
  protected String formatPattern(String pattern) {
    pattern = pattern.trim().replaceAll(" +", " ");
    if (pattern.contains("%d")) {
      if (!pattern.contains("%d{")) {
        pattern = pattern.replaceAll("%d", "%d{yyyy-MM-dd HH:mm:ss}");
      }
    }
    pattern = pattern.replaceAll("%yyyy", "yyyy");
    pattern = pattern.replaceAll("%MMMM", "MMMM");
    pattern = pattern.replaceAll("%MMM", "MMM");
    pattern = pattern.replaceAll("%MM", "MM");
    pattern = pattern.replaceAll("%dd", "dd");

    pattern = pattern.replaceAll("%HH", "HH");
    pattern = pattern.replaceAll("%hh", "hh");
    pattern = pattern.replaceAll("%mm", "mm");
    pattern = pattern.replaceAll("%ss", "ss");

    pattern = pattern.replaceAll("%logger", "%c");
    pattern = pattern.replaceAll("%level", "%p");
    pattern = pattern.replaceAll("%msg", "%m");
    return pattern;
  }

  /**
   * It filters out all log messages with provided filter.
   * 
   * @param appender appender to add filter on.
   * @param filters list of filter which want to apply.
   */
  private void addFilters(Appender appender, ArrayList<LogFilter> filters) {
    try {
      for (LogFilter filterConfig : filters) {
        if (FilterType.RegEx.equals(filterConfig.getName())) {
          appender.addFilter(addLogFilter(filterConfig));
        }
      }
    } catch (Exception e) {
      throw new LoggingException(
          "Exception Occured while adding filter to ::" + getClass().getCanonicalName() + e);
    }
  }

  /**
   * Each Handler can have a filter associated with it. The Handler will call the decide method to
   * check if a given LogRecord should be published. If decide returns 0, the LogRecord will be
   * discarded.
   * 
   * @param config configuration object to be set.
   * @return instance of Filter.
   */
  private Filter addLogFilter(LogFilter config) {
    Filter filter = new Filter() {
      @Override
      public int decide(LoggingEvent log) {
        if (config != null) {
          if (FilterProperty.LOGGER_NAME.equals(config.getApplyOn())
              && log.getLoggerName().matches(config.getExpression())) {
            return onMatch(config.getOnMatch());
          } else if (FilterProperty.MESSAGE.equals(config.getApplyOn())
              && log.getMessage().toString().matches(config.getExpression())) {
            return onMatch(config.getOnMatch());
          }
        }
        return onMismatch(config.getOnMatch());
      }
    };
    return filter;

  }

  /**
   * It convert onMatch value into onMismatch and return the possible replies that a filtering
   * component in log4j can return on mismatch.Ex if onMatch is {@link Filter.DENY} convert into
   * {@link Filter.ACCEPT} and vice versa.
   * 
   * @param onMatch value which want to convert into Filter.
   * @return mismatch value
   */
  private int onMismatch(FilterReply onMatch) {
    return convertFilterReply(onMatch, true);
  }

  /**
   * This return the possible replies that a filtering component in log4j can return.
   * 
   * @param onMatch value which want to convert into FilterReply.
   * @return specific {@link Filter}
   */
  private int onMatch(FilterReply onMatch) {
    return convertFilterReply(onMatch, false);
  }

  /**
   * If invert is true then it will return onMismatch value Ex. if onMatch is {@link Filter.DENY}
   * then it convert into {@link Filter.ACCEPT} and vice versa.Otherwise it return the possible
   * replies that a filtering component in log4j can return.
   * 
   * @param onMatch value which want to convert into FilterReply.
   * @param invert boolean value to identify is onMatch or onMismatch
   * @return specific {@link Filter}
   */
  private int convertFilterReply(FilterReply onMatch, boolean invert) {
    switch (onMatch) {
      case ACCEPT:
        return invert ? Filter.DENY : Filter.ACCEPT;
      case DENY:
        return invert ? Filter.ACCEPT : Filter.DENY;
      default:
        return Filter.NEUTRAL;
    }
  }
}
