/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.logging.logger.config;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import org.appops.logging.logger.config.constant.HandlerAttribute;
import org.appops.logging.logger.config.constant.HandlerType;
import org.appops.logging.meta.Level;

/**
 * An object that maps keys to values.configuration map which required to be used while setup logger
 * handlers.
 *
 * @author deba
 * @version $Id: $Id
 */
public class HandlerConfig {
  private Map<HandlerAttribute, Object> config = new HashMap<HandlerAttribute, Object>();
  private ArrayList<LogFilter> filters;

  /**
   * <p>Constructor for HandlerConfig.</p>
   */
  public HandlerConfig() {
    disable();
  }

  /**
   * <p>addAttribute.</p>
   *
   * @param key a {@link org.appops.logging.logger.config.constant.HandlerAttribute} object.
   * @param value a {@link java.lang.Object} object.
   */
  public void addAttribute(HandlerAttribute key, Object value) {
    getConfig().put(key, value);
  }

  /**
   * <p>Getter for the field <code>config</code>.</p>
   *
   * @return a {@link java.util.Map} object.
   */
  public Map<HandlerAttribute, Object> getConfig() {
    return config;
  }

  /**
   * <p>Setter for the field <code>config</code>.</p>
   *
   * @param configMap a {@link java.util.Map} object.
   */
  public void setConfig(Map<HandlerAttribute, Object> configMap) {
    this.config = configMap;
  }

  /**
   * <p>withName.</p>
   *
   * @param handlerType a {@link org.appops.logging.logger.config.constant.HandlerType} object.
   * @return a {@link org.appops.logging.logger.config.HandlerConfig} object.
   */
  public HandlerConfig withName(HandlerType handlerType) {
    addAttribute(HandlerAttribute.NAME, handlerType);
    return this;
  }

  /**
   * <p>withPattern.</p>
   *
   * @param pattern a {@link java.lang.String} object.
   * @return a {@link org.appops.logging.logger.config.HandlerConfig} object.
   */
  public HandlerConfig withPattern(String pattern) {
    addAttribute(HandlerAttribute.PATTERN, pattern);
    return this;
  }

  /**
   * <p>withLevel.</p>
   *
   * @param level a {@link org.appops.logging.meta.Level} object.
   * @return a {@link org.appops.logging.logger.config.HandlerConfig} object.
   */
  public HandlerConfig withLevel(Level level) {
    addAttribute(HandlerAttribute.LEVEL, level);
    return this;
  }

  /**
   * <p>withFileName.</p>
   *
   * @param filename a {@link java.lang.String} object.
   * @return a {@link org.appops.logging.logger.config.HandlerConfig} object.
   */
  public HandlerConfig withFileName(String filename) {
    addAttribute(HandlerAttribute.FILENAME, filename);
    return this;
  }

  /**
   * <p>withMaxSize.</p>
   *
   * @param size a {@link java.lang.String} object.
   * @return a {@link org.appops.logging.logger.config.HandlerConfig} object.
   */
  public HandlerConfig withMaxSize(String size) {
    addAttribute(HandlerAttribute.MAXSIZE, size);
    return this;
  }

  /**
   * <p>withMaxBackupIndex.</p>
   *
   * @param maxBackupIndex a int.
   * @return a {@link org.appops.logging.logger.config.HandlerConfig} object.
   */
  public HandlerConfig withMaxBackupIndex(int maxBackupIndex) {
    addAttribute(HandlerAttribute.MAX_BACKUP_INDEX, maxBackupIndex);
    return this;
  }

  /**
   * <p>enable.</p>
   *
   * @return a {@link org.appops.logging.logger.config.HandlerConfig} object.
   */
  public HandlerConfig enable() {
    addAttribute(HandlerAttribute.ENABLED, true);
    return this;
  }

  /**
   * <p>disable.</p>
   *
   * @return a {@link org.appops.logging.logger.config.HandlerConfig} object.
   */
  public HandlerConfig disable() {
    addAttribute(HandlerAttribute.ENABLED, false);
    return this;
  }

  /**
   * <p>name.</p>
   *
   * @return a {@link org.appops.logging.logger.config.constant.HandlerType} object.
   */
  public HandlerType name() {
    return HandlerType.valueOf(getConfig().get(HandlerAttribute.NAME).toString());
  }


  /**
   * <p>enabled.</p>
   *
   * @return a boolean.
   */
  public boolean enabled() {
    return (boolean) getConfig().get(HandlerAttribute.ENABLED);
  }

  /**
   * It returns the {@link java.util.ArrayList} of {@link org.appops.logging.logger.config.LogFilter}.
   *
   * @return {@link java.util.ArrayList} of {@link org.appops.logging.logger.config.LogFilter}
   */
  public ArrayList<LogFilter> getFilters() {
    if (filters == null) {
      filters = new ArrayList<LogFilter>();
    }
    return filters;
  }

  /**
   * <p>Setter for the field <code>filters</code>.</p>
   *
   * @param filters a {@link java.util.ArrayList} object.
   */
  public void setFilters(ArrayList<LogFilter> filters) {
    this.filters = filters;
  }

  /**
   * <p>addFilter.</p>
   *
   * @param filter a {@link org.appops.logging.logger.config.LogFilter} object.
   */
  public void addFilter(LogFilter filter) {
    getFilters().add(filter);
  }


}
