/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.logging.meta;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import org.appops.core.constant.IterableConstant;

/**
 * Contains meta data to be used while logging. Contains collection of key-value pair information.
 * Data can be from standard predefined e.g. LEVEL or can be any user defined data.
 *
 * @author deba
 * @version $Id: $Id
 */
public class LogMeta {

  SimpleDateFormat formatter = new SimpleDateFormat("MMM dd,yyyy hh:mm:ss aa");
  private Map<String, Object> metaMap = new HashMap<>();

  /**
   * <p>Constructor for LogMeta.</p>
   */
  public LogMeta() {
    addAttribute(DATETIME.value(), formatter.format(new Date()));
  }

  public static final class DefaultAttribute extends IterableConstant<String> {
    protected DefaultAttribute(String value) {
      super(value);
    }
  }

  /** Constant <code>LEVEL</code> */
  public static final DefaultAttribute LEVEL = new DefaultAttribute("LEVEL");
  /** Constant <code>MESSAGE</code> */
  public static final DefaultAttribute MESSAGE = new DefaultAttribute("MESSAGE");
  /** Constant <code>LOG_TYPE</code> */
  public static final DefaultAttribute LOG_TYPE = new DefaultAttribute("LOG_TYPE");
  /** Constant <code>DATETIME</code> */
  public static final DefaultAttribute DATETIME = new DefaultAttribute("DATETIME");
  /** Constant <code>LOGGERNAME</code> */
  public static final DefaultAttribute LOGGERNAME = new DefaultAttribute("LOGGERNAME");



  /**
   * <p>getAttributeValue.</p>
   *
   * @param key a {@link java.lang.String} object.
   * @return a {@link java.lang.Object} object.
   */
  public Object getAttributeValue(String key) {
    return metaMap.get(key);
  }

  /**
   * Adds meta key-value pair into container.
   *
   * @param key Metadata key. Key can be from standard predefined e.g.
   *        LogMeta.StandardMetaKeys.LEVEL or can be any user defined key.
   * @param value Metadata value.
   * @return Returns updated log meta instance.
   */
  public LogMeta addAttribute(String key, Object value) {
    metaMap.put(key, value);
    return this;
  }


  /**
   * Sets level in log record meta data.
   *
   * @param level Log level.
   * @return Updated log record.
   */
  public LogMeta withLevel(Level level) {
    return addAttribute(LEVEL.value(), level);
  }

  /**
   * Sets log message in log record meta data.
   *
   * @param msg Log message.
   * @return Updated log record.
   */
  public LogMeta withMessage(String msg) {
    return addAttribute(MESSAGE.value(), msg);
  }

  /**
   * <p>Getter for the field <code>metaMap</code>.</p>
   *
   * @return a {@link java.util.Map} object.
   */
  public Map<String, Object> getMetaMap() {
    return metaMap;
  }

  /**
   * <p>Setter for the field <code>metaMap</code>.</p>
   *
   * @param metaMap a {@link java.util.Map} object.
   */
  public void setMetaMap(Map<String, Object> metaMap) {
    this.metaMap = metaMap;
  }

  /**
   * <p>level.</p>
   *
   * @return a {@link java.lang.String} object.
   */
  public String level() {
    return (String) getAttributeValue(LEVEL.value());
  }

  /**
   * <p>message.</p>
   *
   * @return a {@link java.lang.String} object.
   */
  public String message() {
    return String.valueOf(getAttributeValue(MESSAGE.value()));
  }

  /**
   * <p>datetime.</p>
   *
   * @return a {@link java.lang.String} object.
   */
  public String datetime() {
    return (String) getAttributeValue(DATETIME.value());
  }
}
