/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.logging.impl;

import com.google.inject.Inject;
import com.google.inject.name.Named;
import java.util.List;
import org.appops.core.annotation.Config;
import org.appops.logging.destination.DestinationLogger;
import org.appops.logging.destination.DestinationLoggerStore;
import org.appops.logging.logger.Logger;
import org.appops.logging.logger.config.HandlerConfig;
import org.appops.logging.logger.config.LoggerConfig;
import org.appops.logging.logger.config.constant.LoggerType;

/**
 * It manage the logger handlers and formatters and set configuration as per {@link org.appops.logging.logger.config.LoggerConfig}
 * json. and provide appropriate destination logger like {@link java.util.logging.Logger},
 * {@link org.apache.log4j.Logger}}
 *
 * @author deba
 * @version $Id: $Id
 */
public class LogManager {
  private Logger rootLogger;
  private LoggerConfig configuration;
  private DestinationLoggerStore destinationLoggerStore;


  /**
   * Configure loggers as per {@link LoggerConfig} json.
   * 
   * @param configuration Object to be configure.
   */
  private void configure(LoggerConfig configuration) {
    if (configuration != null) {
      List<HandlerConfig> handlers = configuration.getHandlers();
      if (configuration.getDestinationLogger() != null) {
        DestinationLogger<?> destLogger =
            getDestinationLoggerStore().get(configuration.getDestinationLogger());
        destLogger.removeAllHandlers();
        destLogger.addHandlers(handlers);
      }
      if (configuration.getOverrideLoggers() != null) {
        for (LoggerType logger : getDestinationLoggerStore().getDestinationLoggers().keySet()) {
          if (configuration.getOverrideLoggers().contains(logger)
              && !configuration.getDestinationLogger().equals(logger)) {
            DestinationLogger<?> destLogger =
                getDestinationLoggerStore().getDestinationLoggers().get(logger);
            destLogger.removeAllHandlers();
            destLogger.addHandlers(handlers);
          }
        }
      }
    }
  }

  /**
   * <p>getLogger.</p>
   *
   * @param loggerName a {@link java.lang.String} object.
   * @return a {@link org.appops.logging.logger.Logger} object.
   */
  public Logger getLogger(String loggerName) {
    rootLogger.setLoggerName(loggerName);
    return rootLogger;
  }

  /**
   * <p>Getter for the field <code>rootLogger</code>.</p>
   *
   * @return a {@link org.appops.logging.logger.Logger} object.
   */
  public Logger getRootLogger() {
    return rootLogger;
  }

  /**
   * <p>Setter for the field <code>rootLogger</code>.</p>
   *
   * @param rootLogger a {@link org.appops.logging.logger.Logger} object.
   */
  @Inject
  public void setRootLogger(@Named(AppopsRootLogger.NAME) Logger rootLogger) {
    this.rootLogger = rootLogger;
  }

  /**
   * <p>Getter for the field <code>configuration</code>.</p>
   *
   * @return a {@link org.appops.logging.logger.config.LoggerConfig} object.
   */
  public LoggerConfig getConfiguration() {
    return configuration;
  }

  /**
   * A method which configures the given {@link org.appops.logging.logger.config.LoggerConfig} instance and
   * {@link org.appops.logging.destination.DestinationLoggerStore}.
   *
   * @param configuration instance of {@link org.appops.logging.logger.config.LoggerConfig}
   * @param destinationLoggerStore instance of {@link org.appops.logging.destination.DestinationLoggerStore}
   */
  @Inject
  public void init(@Config LoggerConfig configuration,
      DestinationLoggerStore destinationLoggerStore) {
    this.configuration = configuration;
    setDestinationLoggerStore(destinationLoggerStore);
    configure(configuration);


  }

  /**
   * <p>Getter for the field <code>destinationLoggerStore</code>.</p>
   *
   * @return a {@link org.appops.logging.destination.DestinationLoggerStore} object.
   */
  public DestinationLoggerStore getDestinationLoggerStore() {
    return destinationLoggerStore;
  }

  /**
   * <p>Setter for the field <code>destinationLoggerStore</code>.</p>
   *
   * @param destinationLoggerStore a {@link org.appops.logging.destination.DestinationLoggerStore} object.
   */
  public void setDestinationLoggerStore(DestinationLoggerStore destinationLoggerStore) {
    this.destinationLoggerStore = destinationLoggerStore;
  }
}
