/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.logging.logger.store;

import com.google.inject.Inject;
import java.io.File;
import java.io.FileWriter;
import java.io.PrintWriter;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import org.appops.core.annotation.Config;
import org.appops.logging.logger.config.LoggerConfig;
import org.appops.logging.meta.Level;
import org.appops.logging.meta.LogMeta;
import org.appops.logging.meta.LogMetaList;
import org.appops.marshaller.DescriptorType;
import org.appops.marshaller.Marshaller;

/**
 * <p>
 * FileLogStoreImpl class.
 * </p>
 *
 * @author deba
 * @version $Id: $Id
 */
public class FileLogStoreImpl implements LogStore {
  private Marshaller marshaller;
  private LoggerConfig config;

  /** {@inheritDoc} */
  @Override
  public void addLogRecord(LogMeta logRecord) {
    String logMetaJson = getMarshaller().marshall(logRecord, getMarshallerType());
    saveMeta(logMetaJson);
  }

  private void saveMeta(String meta) {
    try {
      String path = getStorageFilePath();
      File file = new File(path);
      if (path.contains(File.separator)) {
        file.getParentFile().mkdirs();
      }
      if (getMaxFileSize(getConfig().getStorage().getMaxFileSize()) > file.length()) {
        FileWriter writer = new FileWriter(file, true);
        writer.write(meta + ",");
        writer.close();
      } else {
        removeOlderLogs();
      }
    } catch (Exception e) {
      System.out.println(e);
    }
  }

  /**
   * {@inheritDoc}
   *
   * Fetch all logs from file store.
   */
  @Override
  public Collection<LogMeta> getAllLogRecords() {
    try {
      String data = readDataFromFile();
      if (data != null) {
        String allLogs = formatLogs(data);
        List<LogMeta> returnMetaList =
            getMarshaller().unmarshall(allLogs, LogMetaList.class, getMarshallerType());
        return returnMetaList;
      }
    } catch (Exception e) {
      System.out.println("Exception occured while fetching logs from store " + e);
    }
    return null;
  }


  /**
   * Remove older logs from store.
   */
  public void removeOlderLogs() {
    try {
      Collection<LogMeta> logs = getAllLogRecords();
      Collection<LogMeta> allLogs = new ArrayList<LogMeta>(logs);
      for (LogMeta logMeta : allLogs) {
        if (!isValid(logMeta.datetime())) {
          logs.remove(logMeta);
        }
      }
      String filterdLogs = getMarshaller().marshall(logs, DescriptorType.JSON);
      filterdLogs = filterdLogs.substring(1, filterdLogs.length() - 1);
      new PrintWriter(getStorageFilePath()).close();
      saveMeta(filterdLogs);
    } catch (Exception e) {
      System.out.println("Exception occured while removing older logs from store " + e);
    }
  }

  /**
   * Compare dates and check whether the datetime is after TTL.
   * 
   * @param datetime date which to be compare.
   * @return true if datetime is after TTL otherwise false;
   */
  private boolean isValid(String datetime) {
    DateFormat format = new SimpleDateFormat("MMM dd,yyyy", Locale.ENGLISH);
    try {
      Date metaDate = format.parse(datetime);
      Date ttl = format.parse(getTtl());
      if (metaDate.after(ttl)) {
        return true;
      }
    } catch (ParseException e) {
      e.printStackTrace();
    }
    return false;
  }

  /**
   * Read all logs from file.
   * 
   * @return string of all logs.
   */
  private String readDataFromFile() {
    try {
      String data = new String(Files.readAllBytes(Paths.get(getStorageFilePath())));
      return data;
    } catch (Exception e) {
      System.out.println(
          "Exception occured in " + getClass().getCanonicalName() + ":readDataFromFile() " + e);
    }
    return null;
  }


  /**
   * Format data string like remove last comma.
   * 
   * @param data to be format
   * @return formatted data.
   */
  private String formatLogs(String data) {
    if (data != null && !data.isEmpty()) {
      if (data.endsWith(",")) {
        data = data.substring(0, data.length() - 1);
      }
      data = "[" + data + "]";
      return data;

    }
    return null;
  }

  /**
   * Fetch TTl from {@link LoggerConfig}.
   * 
   * @return ttl date string
   */
  private String getTtl() {
    return getConfig().getStorage().getTtl();
  }

  /**
   * Create and return storage file path.
   * 
   * @return storage file path
   */
  private String getStorageFilePath() {
    return getConfig().getStorage().getFilepath();
  }

  /**
   * Fetch {@link MarshallerType} from file extension .
   * 
   * @return {@link MarshallerType}.
   */
  private DescriptorType getMarshallerType() {
    try {
      if (getStorageFilePath() != null) {
        return DescriptorType.fromExtension(getStorageFilePath());
      }
    } catch (Exception e) {
      System.out.println(e);
    }
    return DescriptorType.JSON;
  }

  /**
   * Convert file size from KB/MB/GB into bytes.
   * 
   * @param maxFileSizeString size which is to be convert into byte
   * @return byte representation of provided file size.
   */
  private long getMaxFileSize(String maxFileSizeString) {
    int defaultSize = 100 * 1024 * 1024;
    try {
      if (maxFileSizeString != null) {
        String unit =
            maxFileSizeString.substring(maxFileSizeString.length() - 2, maxFileSizeString.length());
        int size = Integer.parseInt(maxFileSizeString.substring(0, maxFileSizeString.length() - 2));

        switch (unit.toUpperCase()) {
          case "KB":
            return size * 1024;
          case "MB":
            return size * 1024 * 1024;
          case "GB":
            if (size <= 2) {
              return size * 1024 * 1024 * 1024;
            } else {
              return defaultSize;
            }
          default:
            return defaultSize;
        }
      }
    } catch (Exception e) {
      System.out
          .println("Exception occured while converting file size it will use default size." + e);
    }
    return defaultSize;
  }

  /**
   * {@inheritDoc}
   *
   * Filter log records by keyword.
   */
  @Override
  public Collection<LogMeta> filterByKeyword(String keyword) {
    List<LogMeta> listToReturn = new ArrayList<>();
    for (LogMeta logRecord : getAllLogRecords()) {
      Map<String, Object> logMetaMap = logRecord.getMetaMap();
      for (Object value : logMetaMap.values()) {
        if (String.valueOf(value).contains(keyword)) {
          listToReturn.add(logRecord);
        }
      }
    }
    return listToReturn;
  }


  /**
   * {@inheritDoc}
   *
   * Filter log records by level.
   */
  // TODO level WARN-WARNING handling
  @Override
  public Collection<LogMeta> filterByLevel(Level expectedLevel) {
    List<LogMeta> listToReturn = new ArrayList<>();
    if (expectedLevel == null) {
      return listToReturn;
    }
    for (LogMeta logRecord : getAllLogRecords()) {
      String level = (String) logRecord.getMetaMap().get(LogMeta.LEVEL.value());
      if (expectedLevel.name().equals(level) || expectedLevel.name().equals(Level.ALL.name())) {
        listToReturn.add(logRecord);
      }
    }
    return listToReturn;
  }

  /** {@inheritDoc} */
  @Override
  public Collection<LogMeta> getLogRecordsByPage(Integer startIndex, Integer pageSize) {
    // TODO Auto-generated method stub
    return null;
  }

  /**
   * {@inheritDoc}
   *
   * Get filtered log records by start date.
   */
  @Override
  public Collection<LogMeta> getLogRecordsByStartDate(String startDateOfLog) {
    List<LogMeta> listToReturn = new ArrayList<>();
    Date startDate = null;
    startDateOfLog = startDateOfLog.substring(0, 9);

    DateFormat format = new SimpleDateFormat("MMMddyyyy", Locale.ENGLISH);
    try {
      startDate = format.parse(startDateOfLog);
    } catch (ParseException e) {
      e.printStackTrace();
    }
    for (LogMeta logRecord : getAllLogRecords()) {
      Date date2;
      String dateValue = (String) logRecord.datetime();
      dateValue = dateValue.substring(0, 12);
      dateValue = dateValue.replace(",", "");

      DateFormat format2 = new SimpleDateFormat("MMM ddyyyy", Locale.ENGLISH);
      try {
        date2 = format2.parse(dateValue);
        if (date2.compareTo(startDate) >= 0) {
          listToReturn.add(logRecord);
        }
      } catch (ParseException e) {
        e.printStackTrace();
      }
    }

    return listToReturn;
  }

  /**
   * {@inheritDoc}
   *
   * Get filtered log records by start date and end date.
   */
  @Override
  public Collection<LogMeta> getLogRecords(String startDate, String endDate) {
    return filterByDateTime(startDate, endDate, null, null);
  }

  /**
   * {@inheritDoc}
   *
   * Get filtered log records by start time.
   */
  @Override
  public Collection<LogMeta> getLogRecordsByStartTime(String startTime) {
    return filterByDateTime(null, null, startTime, null);
  }

  /**
   * {@inheritDoc}
   *
   * Get filtered log records by start time and end time.
   */
  @Override
  public Collection<LogMeta> getLogRecordsByStartTimeEndTime(String startTime, String endTime) {
    return filterByDateTime(null, null, startTime, endTime);
  }



  /**
   * {@inheritDoc}
   *
   * Get filtered log records by date and time. The method invokes the
   * getFilteredDateListOfDateTime() method for date filtration. After that invokes
   * getFinalFilteredListOfDateTime() for the time filtration from the result of filtered date list.
   */
  @Override
  public Collection<LogMeta> getLogRecordsByDateAndTime(String startDate, String endDate,
      String startTime, String endTime) {
    return filterByDateTime(startDate, endDate, startTime, endTime);


  }

  /**
   * Get filtered log records by provided date and time,if end time or end date is null then it will
   * use current date and time.
   * 
   * @param startDate : Starting Date of Log
   * @param endDate : Ending Date of Log
   * @param startTime : Starting Time of Log
   * @param endTime : Ending Time of Log
   * @return Collection of filtered LogMeta records.
   */
  private Collection<LogMeta> filterByDateTime(String startDateString, String endDateString,
      String startTime, String endTime) {
    List<LogMeta> listToReturn = new ArrayList<>();
    Date startDate = null;
    Date endDate = null;
    DateFormat formatter = new SimpleDateFormat("MMMddyyyyhh:mm", Locale.ENGLISH);
    try {
      if (startDateString != null) {
        startDateString = getStartDate(startDateString, startTime);;
        startDate = formatter.parse(startDateString);
      }
      endDateString = getEndString(endDateString, endTime);
      endDate = formatter.parse(endDateString);

    } catch (ParseException e) {
      e.printStackTrace();
    }
    DateFormat format2 = new SimpleDateFormat("MMM dd,yyyy hh:mm:ss aa", Locale.ENGLISH);

    for (LogMeta logRecord : getAllLogRecords()) {
      try {
        Date logMetaDate = format2.parse(logRecord.datetime());

        if (logMetaDate.compareTo(startDate) >= 0 && logMetaDate.compareTo(endDate) <= 0) {
          listToReturn.add(logRecord);
        }
      } catch (ParseException e) {
        e.printStackTrace();
      }
    }
    return listToReturn;

  }

  /**
   * Create date string from provided date and time.
   * 
   * @param endDateString date string
   * @param endTime time
   * @return populated date string.
   */
  private String getEndString(String endDateString, String endTime) {
    LocalTime endLocalTime = null;
    if (endDateString == null || endDateString.isEmpty()) {
      endDateString = new SimpleDateFormat("MMMddyyyy").format(new Date());
    } else {
      endDateString = endDateString.substring(0, 9);
    }
    if (endTime == null || endTime.isEmpty()) {
      endLocalTime = LocalTime.now();
    } else {
      endLocalTime = LocalTime.parse(endTime);
    }
    return endDateString + endLocalTime;
  }

  /**
   * Create date string from provided date and time.
   * 
   * @param endDateString date string
   * @param endTime time
   * @return populated date string.
   */
  private String getStartDate(String startDateString, String startTime) {
    startDateString = startDateString.substring(0, 9);
    if (startTime != null && !startTime.isEmpty()) {
      startDateString = startDateString + startTime;
    } else {
      startDateString = startDateString + "00:00";
    }
    return startDateString;
  }

  /**
   * <p>
   * Getter for the field <code>marshaller</code>.
   * </p>
   *
   * @return a {@link org.appops.marshaller.Marshaller} object.
   */
  public Marshaller getMarshaller() {
    return marshaller;
  }


  /**
   * <p>
   * Setter for the field <code>marshaller</code>.
   * </p>
   *
   * @param marshaller a {@link org.appops.marshaller.Marshaller} object.
   */
  @Inject
  public void setMarshaller(Marshaller marshaller) {
    this.marshaller = marshaller;
  }

  /**
   * <p>
   * Getter for the field <code>config</code>.
   * </p>
   *
   * @return a {@link org.appops.logging.logger.config.LoggerConfig} object.
   */
  public LoggerConfig getConfig() {
    return config;
  }

  /**
   * <p>
   * Setter for the field <code>config</code>.
   * </p>
   *
   * @param config a {@link org.appops.logging.logger.config.LoggerConfig} object.
   */
  @Inject
  public void setConfig(@Config LoggerConfig config) {
    this.config = config;
  }

}
