/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.maven.plugin.mojo;

import com.google.inject.Guice;
import com.google.inject.Injector;
import java.io.File;
import java.net.URLClassLoader;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import org.apache.commons.io.FileUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;
import org.appops.configuration.guice.ConfigServiceModule;
import org.appops.core.deployment.ServiceConfiguration;
import org.appops.logging.guice.DefaultLoggerModule;
import org.appops.marshaller.DescriptorType;
import org.appops.marshaller.Marshaller;
import org.appops.marshaller.guice.MarshallerModule;
import org.appops.maven.plugin.helper.MojoHelper;
import org.appops.service.deployment.ContainerType;
import org.appops.service.deployment.docker.DockerDeploymentManager;
import org.appops.service.exception.DeploymentException;

/**
 * Appops maven plugin which deploys appops service application using configuration parameters
 * provided.
 */
@Mojo(name = "deploy-app", requiresDependencyResolution = ResolutionScope.COMPILE_PLUS_RUNTIME)
public class ServiceDeploymentMojo extends AbstractMojo {

  @Parameter(required = true)
  private ContainerType containerType;
  @Parameter(required = true)
  private String profileRoot;
  @Parameter(defaultValue = "${project}")
  private MavenProject project;


  @Override
  public void execute() throws MojoExecutionException, MojoFailureException {
    try {

      URLClassLoader classLoader = new MojoHelper().getClassLoader(project);
      Thread.currentThread().setContextClassLoader(classLoader);
      classLoader.loadClass("org.appops.service.impl.guice.ServiceStoreModule");

      File configFile = new File(profileRoot + "core.yml");
      if (!configFile.exists()) {

        throw new DeploymentException("Config file not found ->" + configFile);
      }

      performDeployment(configFile);
    } catch (Exception e) {

      e.printStackTrace();
    }
  }

  /**
   * Reads configuration file and deploys service application.
   * 
   * @param configFile Deployment configuration file.
   */
  private void performDeployment(File configFile) {
    if (ContainerType.DOCKER.equals(getContainerType())) {

      performDockerDeployment(configFile);
    } else {

      throw new DeploymentException("Please provide valid container-type, supported types are - "
          + Arrays.toString(ContainerType.values()));
    }
  }

  /**
   * Reads configuration file and deploys service application into docker based enviroment.
   * 
   * @param configFile Docker configuration file.
   */
  private void performDockerDeployment(File configFile) {

    try {

      Injector systemInjector = createSystemInjector();

      String configString = FileUtils.readFileToString(configFile, StandardCharsets.UTF_8);
      Marshaller marshaller = systemInjector.getInstance(Marshaller.class);
      ServiceConfiguration config = marshaller.unmarshall(configString, ServiceConfiguration.class,
          DescriptorType.fromExtension(configFile.getName()));

      DockerDeploymentManager deploymentManager =
          systemInjector.getInstance(DockerDeploymentManager.class);

      deploymentManager.performDeployment(config);
      System.out.println("Docker deployment completed.");
    } catch (Exception e) {

      throw new DeploymentException(e);
    }
  }

  public Injector createSystemInjector() {
    return Guice.createInjector(new ConfigServiceModule(), new MarshallerModule(),
        new DefaultLoggerModule());
  }


  public ContainerType getContainerType() {
    return containerType;
  }

  public void setContainerType(ContainerType containerType) {
    this.containerType = containerType;
  }

}
