/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.maven.plugin.mojo;

import java.io.File;
import java.lang.annotation.Annotation;
import java.net.URISyntaxException;
import java.net.URLClassLoader;
import java.util.HashSet;
import java.util.Set;
import org.apache.commons.io.FilenameUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;
import org.appops.core.ClassPathAnalyser;
import org.appops.core.TypeScanner;
import org.appops.maven.plugin.helper.MojoHelper;
import org.appops.tsgen.TypeScriptGenerator;

/**
 * Scans service interfaces using service annotation provided and generate typescript module
 * definition for service.
 */
@Mojo(name = "generate-ts", requiresDependencyResolution = ResolutionScope.RUNTIME)
public class TypeScriptGeneratorMojo extends AbstractMojo {



  @Parameter(required = true)
  private String serviceAnnotation;

  @Parameter(required = true)
  private File outputDir;

  @Parameter(defaultValue = "${project}", required = true)
  private MavenProject project;

  /**
   * Directory containing the classes and resource files that should be packaged into the JAR.
   */
  @Parameter(defaultValue = "${project.build.outputDirectory}", required = true)
  private File classesDirectory;

  @Override
  public void execute() throws MojoExecutionException, MojoFailureException {

    try {

      URLClassLoader classLoader = new MojoHelper().getClassLoader(project);
      Thread.currentThread().setContextClassLoader(classLoader);
      ClassPathAnalyser classPathAnalyser = new ClassPathAnalyser(classLoader);
      classPathAnalyser.setTypeScanner(new TypeScanner());

      // URLClassLoader classLoader = new MojoHelper().getClassLoader(project);
      Class<? extends Annotation> annotation =
          (Class<? extends Annotation>) classLoader.loadClass(serviceAnnotation);
      Set<Class<?>> classes = new HashSet<>();
      Set<Class<?>> annotatedTypes = classPathAnalyser.getAnnotatedTypes(annotation);

      for (Class<?> clazz : annotatedTypes) {
        if (clazz.isInterface() && clazz.isAnnotationPresent(annotation)) {

          classes.add(clazz);
        }
      }


      // findAnnotatedTypes(classesDirectory.listFiles(), classLoader, annotation, classes);

      TypeScriptGenerator tsGenerator = new TypeScriptGenerator();
      tsGenerator.generateTypeScript(classes, annotation, outputDir);
    } catch (Exception e) {

      e.printStackTrace();
    }

  }



  /**
   * Filters out types annotated with service annotation provided.
   * 
   * @param files List of resource files obtained from project output directory.
   * @param loader Custom url class loader.
   * @param annotation Service annotation class.
   * @param annotatedTypes Set into which located annotated types are to be added.
   */
  private void findAnnotatedTypes(File[] files, URLClassLoader loader,
      Class<? extends Annotation> annotation, Set<Class<?>> annotatedTypes)
      throws URISyntaxException, ClassNotFoundException {

    for (File resource : files) {

      if (resource.isDirectory()) {

        findAnnotatedTypes(resource.listFiles(), loader, annotation, annotatedTypes);
      } else if ("class".equals(FilenameUtils.getExtension(resource.getName()))) {

        String path = resource.getParentFile().getAbsolutePath() + "/"
            + FilenameUtils.getBaseName(resource.getName());
        path = path.replace(classesDirectory.getAbsolutePath(), "");
        path = path.startsWith("/") ? path.substring(1) : path;
        String className = path.replaceAll("/", "\\.");

        Class<?> clazz = loader.loadClass(className);
        if (clazz.isInterface() && clazz.isAnnotationPresent(annotation)) {

          annotatedTypes.add(clazz);
        }
      }
    }
  }

}
