/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */
package org.appops.maven.plugin.mojo;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Collection;
import java.util.Iterator;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.apache.commons.io.IOUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;
import org.appops.core.ClassPathAnalyser;
import org.appops.core.TypeScanner;
import org.appops.core.service.annotation.Service;
import org.appops.maven.plugin.helper.MojoHelper;
import org.appops.service.ServiceInitializer;

@Mojo(name = "clubbed-webapp", requiresDependencyResolution = ResolutionScope.COMPILE_PLUS_RUNTIME)
public class ClubbedWebappContentMojo extends AbstractMojo {

  @Parameter(defaultValue = "${project}")
  private MavenProject project;
  @Parameter(required = true)
  private String webappPath;
  private Logger logger = Logger.getLogger(this.getClass().getCanonicalName());


  @Override
  public void execute() throws MojoExecutionException, MojoFailureException {
    Set<Artifact> artifacts = project.getArtifacts();
    Iterator<Artifact> iterator = artifacts.iterator();

    while (iterator.hasNext()) {
      String serviceName = "";
      Artifact artifact = iterator.next();
      if (isImplDependency(artifact)) {
        File artifactFile = artifact.getFile();
        serviceName = getServiceName(artifactFile);
        if (artifactFile.exists() && serviceName != null && !serviceName.isEmpty()) {
          copyWebappContent(artifactFile, serviceName);
        }
      }
    }
  }

  /**
   * It return service name from given artifact file.
   * 
   * @param artifactFile dependency as a artifact file
   * @return service name
   */
  private String getServiceName(File artifactFile) {

    ClassPathAnalyser projectClassPathAnalyser = null;
    URLClassLoader projectClassLoader = null;
    try {
      projectClassLoader = new MojoHelper().getClassLoader(project);
    } catch (Exception e1) {
      logger.log(Level.SEVERE, "Exception occurred while creating classloader");
    }

    Thread.currentThread().setContextClassLoader(projectClassLoader);
    projectClassPathAnalyser = new ClassPathAnalyser(projectClassLoader);
    projectClassPathAnalyser.setTypeScanner(new TypeScanner());

    Collection<Class<? extends ServiceInitializer>> projectInitializers =
        projectClassPathAnalyser.subTypesOf(ServiceInitializer.class);

    return getServiceNameFromInitializer(projectInitializers, artifactFile);

  }

  /**
   * It return service name from given artifact file.
   * 
   * @param projectInitializers initializers which are found in project through reflection
   * @param artifactFile dependency as a artifact file
   * @return service name from initializer
   */
  private String getServiceNameFromInitializer(
      Collection<Class<? extends ServiceInitializer>> projectInitializers, File artifactFile) {

    Class<? extends ServiceInitializer> artifactInitializer = getArtifactInitializer(artifactFile);

    String serviceName = "";
    for (Class<? extends ServiceInitializer> projectInitializer : projectInitializers) {

      String projectInitializerName = projectInitializer.getCanonicalName();
      String artifactInitializerName = artifactInitializer.getCanonicalName();

      if (projectInitializerName.equals(artifactInitializerName)) {
        Annotation[] annotationsOnInitializer = projectInitializer.getAnnotations();

        for (Annotation annotation : annotationsOnInitializer) {
          Class<? extends Annotation> serviceAnnotation = annotation.annotationType();
          Annotation[] annotationsOnAnnotation = serviceAnnotation.getAnnotations();

          for (Annotation annotation2 : annotationsOnAnnotation) {
            if (annotation2.annotationType().equals(Service.class)) {
              serviceName = serviceAnnotation.getSimpleName();
              return serviceName;
            }
          }
        }
      }
    }
    return serviceName;
  }

  /**
   * It return initializer from given artifact file.
   * 
   * @param artifactFile dependency as a artifact file
   * @return initializer from given artifact file
   */
  private Class<? extends ServiceInitializer> getArtifactInitializer(File artifactFile) {

    ClassPathAnalyser artifactClassPathAnalyser = null;
    URLClassLoader artifactClassloader = null;

    try {
      artifactClassloader = new URLClassLoader(new URL[] {artifactFile.toURI().toURL()},
          this.getClass().getClassLoader());
    } catch (Exception e1) {
      logger.log(Level.SEVERE, "Exception occurred while creating classloader");
    }

    Thread.currentThread().setContextClassLoader(artifactClassloader);
    artifactClassPathAnalyser = new ClassPathAnalyser(artifactClassloader);
    artifactClassPathAnalyser.setTypeScanner(new TypeScanner());

    Collection<Class<? extends ServiceInitializer>> artifactInitializers =
        artifactClassPathAnalyser.subTypesOf(ServiceInitializer.class);

    int totalInitializers = artifactInitializers.size();
    if (totalInitializers == 1) {
      return artifactInitializers.iterator().next();
    }

    return null;
  }

  /**
   * It copies all webapp content into current project webapp folder under given service name
   * folder.
   * 
   * @param artifactFile dependency as a artifact file
   * @param serviceName service name
   */
  private void copyWebappContent(File artifactFile, String serviceName) {
    try {
      ZipInputStream stream =
          new ZipInputStream(new BufferedInputStream(new FileInputStream(artifactFile)));
      ZipEntry entry = null;
      while ((entry = stream.getNextEntry()) != null) {
        String entryName = entry.getName();
        if (entryName.startsWith("webapp")) {
          File newFile = null;
          if (entryName.endsWith("/")) {
            createDir(entryName, newFile, serviceName);
          } else {
            copyFileIntoDir(entryName, newFile, stream, serviceName);
          }
        }
      }
    } catch (Exception e) {
      logger.log(Level.SEVERE, "Exception occurred while iterating files from jar");
    }
  }

  /**
   * It copies the the file into given directory location.
   * 
   * @param entryName file name
   * @param newFile new file which contains data
   * @param stream contains data wants to copy
   * @param serviceName service name
   */
  private void copyFileIntoDir(String entryName, File newFile, ZipInputStream stream,
      String serviceName) {
    entryName = entryName.replace("webapp/", "");
    newFile = new File(webappPath + "/" + serviceName + "/", entryName);
    try (FileOutputStream fis = new FileOutputStream(newFile);) {
      IOUtils.copy(stream, fis);
    } catch (IOException e) {
      logger.log(Level.SEVERE, "Exception occurred while copying file");
    }

  }

  /**
   * It creates the directory under specific service name.
   * 
   * @param entryName name of the file
   * @param newFile file wants create
   * @param serviceName name of the service
   */
  private void createDir(String entryName, File newFile, String serviceName) {
    entryName = entryName.replace("webapp/", "");
    newFile = new File(webappPath + "/" + serviceName + "/", entryName);
    newFile.mkdirs();
  }

  /**
   * It checks given artifact is impl dependency or not and return appropriate result.
   * 
   * @param artifact dependency as an artifact
   * @return true, if impl dependency otherwise false
   */
  private boolean isImplDependency(Artifact artifact) {
    String artifactId = artifact.getArtifactId();
    String groupId = artifact.getGroupId();

    if ((groupId.equals("com.ainosoft") || groupId.equals("org.appops"))
        && (artifactId.endsWith("-impl") || artifactId.equals("appops-br-service-store"))) {
      return true;
    }
    return false;
  }

}
