/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.scheduler.queue;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import org.appops.core.job.token.JobToken;
import org.appops.core.job.token.TokenType;
import org.appops.job.JobTokenQueue;
import org.appops.scheduler.exception.SchedulerException;
import org.quartz.CronExpression;

/**
 * Default implementation of {@link org.appops.job.JobTokenQueue}.
 *
 * @author deba
 * @version $Id: $Id
 */
public class JobTokenQueueImpl implements JobTokenQueue {

  private List<JobToken> tokenQueue = Collections.synchronizedList(new LinkedList<>());
  private SortedMap<Date, ArrayList<JobToken>> scheduleQueue =
      Collections.synchronizedSortedMap(new TreeMap<>());

  /** {@inheritDoc} */
  @Override
  public Set<JobToken> popTokens() {
    Set<JobToken> selectedTokens = new HashSet<JobToken>();
    selectedTokens.addAll(getElapsedSchedules());
    JobToken jobToken = getNextJobToken();
    if (jobToken != null) {
      selectedTokens.add(jobToken);
    }
    return selectedTokens;
  }

  /** {@inheritDoc} */
  @Override
  public void addToken(JobToken token) {
    try {
      if (TokenType.SCHEDULE.equals(token.getTokenType())) {
        addScheduleToken(token);
      } else {
        tokenQueue.add(token);
      }
    } catch (Exception e) {
      throw new SchedulerException("Unabled to add token to queue", e);
    }
  }


  private void addScheduleToken(JobToken token) {
    try {
      Date schedule = getNextValidTimeFor(token.getScheduleExpression());
      getScheduledTokens(schedule).add(token);
    } catch (Exception e) {
      throw new SchedulerException("Unabled to add token to queue", e);
    }
  }


  /**
   * Get next Job token.
   *
   * @return JobToken.
   */
  public JobToken getNextJobToken() {
    if (tokenQueue.isEmpty()) {
      return null;
    }

    JobToken token = tokenQueue.get(0);
    tokenQueue.remove(token);
    return token;

  }

  /**
   * Get next Elapsed Schedules.
   *
   * @return Set of ScheduleToken .
   */
  // TODO: This method might contain a bug, need to be checked/tested.
  public Set<JobToken> getElapsedSchedules() {
    Date currentTime = new Date();
    Set<JobToken> selectedSchedules = new HashSet<>();

    while (new TreeMap<>(scheduleQueue).lowerKey(currentTime) != null) {
      Date elapsedKey = new TreeMap<>(scheduleQueue).lowerKey(currentTime);

      ArrayList<JobToken> elapsedSchedules = scheduleQueue.get(elapsedKey);
      selectedSchedules.addAll(elapsedSchedules);
      elapsedSchedules.removeAll(selectedSchedules);
      if (elapsedSchedules.isEmpty()) {
        scheduleQueue.remove(elapsedKey);
      }
    }
    return selectedSchedules;
  }

  /**
   * Calculates next valid time for schedule token.
   *
   * @param scheduleExpression Chron expression.
   * @return Next valid time for cron expression provided.
   * @throws java.text.ParseException if token information is not parsable.
   */
  protected Date getNextValidTimeFor(String scheduleExpression) throws ParseException {
    CronExpression cronExpression = new CronExpression(scheduleExpression);
    return cronExpression.getNextValidTimeAfter(new Date());
  }


  /**
   * Fetches tokens for schedule provided.
   * 
   * @param schedule Schedule for which tokens are to be fetched.
   * @return List of schedule token configured for schedule provided.
   */
  private ArrayList<JobToken> getScheduledTokens(Date schedule) {
    if (!scheduleQueue.containsKey(schedule) || scheduleQueue.get(schedule) == null) {
      scheduleQueue.put(schedule, new ArrayList<JobToken>());
    }
    return scheduleQueue.get(schedule);
  }


  /** {@inheritDoc} */
  @Override
  public Boolean removeJobTokens(List<JobToken> tokens) {
    return tokenQueue.removeAll(tokens);
  }

  /** {@inheritDoc} */
  @Override
  public Map<Date, ArrayList<JobToken>> getScheduleTokens() {
    Map<Date, ArrayList<JobToken>> map = new HashMap<>();
    map.putAll(scheduleQueue);
    return map;
  }


  /** {@inheritDoc} */
  @Override
  public Boolean isTokenPresent(JobToken token) {
    return tokenQueue.contains(token) || ((TokenType.SCHEDULE.equals(token.getTokenType()))
        && isScheduledTokenPresent((JobToken) token));
  }


  private Boolean isScheduledTokenPresent(JobToken token) {
    for (Collection<JobToken> tokens : scheduleQueue.values()) {
      if (tokens.contains(token)) {
        return true;
      }
    }
    return false;
  }

  /** {@inheritDoc} */
  @Override
  public Integer size() {
    int size = tokenQueue.size();
    for (Collection<JobToken> tokens : scheduleQueue.values()) {
      size += tokens.size();
    }
    return size;
  }

}
