/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.scheduler.thread;

import com.google.inject.Inject;
import java.util.Timer;
import java.util.TimerTask;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.appops.core.annotation.Config;
import org.appops.job.JobPipelineProcessor;
import org.appops.job.JobSchedulerThread;
import org.appops.job.TokenProcessor;
import org.appops.scheduler.config.SchedulerConfig;

/**
 * A task that can be scheduled for one-time or repeated execution by a Timer.
 *
 * @author prashant
 * @version $Id: $Id
 */
public class JobSchedulerThreadImpl implements JobSchedulerThread {

  private TokenProcessor tokenProcessor;
  private JobPipelineProcessor jobPipelineProcessor;
  private Logger logger = Logger.getLogger(getClass().getCanonicalName());
  private Timer timer;
  private long period;

  @Inject
  /**
   * <p>
   * Constructor for JobSchedulerThreadImpl.
   * </p>
   *
   * @param tokenProcessor a {@link org.appops.job.TokenProcessor} object.
   * @param jobPipelineProcessor a {@link org.appops.job.JobPipelineProcessor} object.
   */
  public JobSchedulerThreadImpl(TokenProcessor tokenProcessor,
      JobPipelineProcessor jobPipelineProcessor) {
    this.tokenProcessor = tokenProcessor;
    this.jobPipelineProcessor = jobPipelineProcessor;
  }

  /**
   * Initialize period with provided configuration value if it is null then it will used default
   * value i.e 1000(1 Second).
   *
   * @param schedulerConfig configuration object which provides needed configuration.
   */
  @Inject
  public void initSchedulerConfig(@Config SchedulerConfig schedulerConfig) {
    period = schedulerConfig != null && schedulerConfig.getTokenPollPeriodInSec() != null
        ? schedulerConfig.getTokenPollPeriodInSec() * 1000
        : 1000;
  }



  /**
   * {@inheritDoc}
   *
   * Schedules the specified task for repeated fixed-delay execution, beginning after the specified
   * delay. Subsequent executions take place at approximately regular intervals separated by the
   * specified period.
   */
  @Override
  public void startJobScheduler() {
    timer = new Timer();
    timer.schedule(createTimerTask(), 0, period);
  }



  private TimerTask createTimerTask() {
    return new TimerTask() {
      @Override
      public void run() {
        try {
          jobPipelineProcessor.processJobPipeline();
          tokenProcessor.processTokens();
        } catch (Exception e) {
          // logger.log(Level.SEVERE, "" + e);
        }
      }
    };
  }

  /**
   * Terminates this timer, discarding any currently scheduled tasks. Does not interfere with a
   * currently executing task (if it exists). Once a timer has been terminated, its execution thread
   * terminates gracefully, and no more tasks may be scheduled on it.
   */
  public void stop() {
    if (timer != null) {
      timer.cancel();
      logger.log(Level.INFO, "Timer is terminated,It Does not interfere with a\n"
          + "   * currently executing task (if it exists)");
    }
  }



}
