/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.scheduler.executor;

import org.appops.core.ClassPathAnalyser;
import org.appops.core.job.token.JobToken;
import org.appops.core.job.token.JobTokenInputInfo;
import org.appops.job.JobPipeline;
import org.appops.job.JobPipelineIoConverter;
import org.appops.job.JobPipelineQueue;
import org.appops.job.JobStage;
import org.appops.job.JobStageConverter;
import org.appops.log.service.slim.service.Level;
import org.appops.logging.impl.LogManager;
import org.appops.logging.logger.Logger;
import org.appops.marshaller.DescriptorType;
import org.appops.marshaller.Marshaller;
import org.appops.service.event.JobEvent;
import org.appops.service.job.TokenRouter;
import org.appops.slim.base.api.ServiceMetaManager;
import org.quartz.SchedulerException;
import com.google.inject.Inject;
import com.google.inject.Provider;

/**
 * Execute {@link org.appops.core.job.token.JobToken} from {@link org.appops.job.JobPipeline}.
 *
 * @author deba
 * @version $Id: $Id
 */
public class JobPipelineExecutorImpl implements JobPipelineExecutor {

  @Inject
  private Provider<TokenRouter> tokenRouterProvider;
  private JobPipelineQueue jobPipelineQueue;
  private ClassPathAnalyser classPathAnalyser;
  private Logger logger;
  @Inject
  private Provider<JobStageConverter> jobStageConverter;
  @Inject
  DerivedIoConverterFactory derivedIoConverterFactory;
  @Inject
  private Marshaller marshaller;
  private Provider<ServiceMetaManager> serviceMetaManager;

  /**
   * Default constructor.
   */
  public JobPipelineExecutorImpl() {

  }


  /**
   * <p>
   * Constructor for JobPipelineExecutorImpl.
   * </p>
   *
   * @param logManager a {@link org.appops.logging.impl.LogManager} object.
   */
  @Inject
  public JobPipelineExecutorImpl(LogManager logManager) {
    logger = logManager.getRootLogger();
  }

  /**
   * Execute next job token in the {@link JobPipeline}.
   */
  @Override
  public void executeNextJobToken(JobEvent event) {
    if (JobEvent.STARTED.equals(event.getEventStatus())) {
      JobToken startedJobToken = convertJobTokenJson(event.jobToken());
      jobStartedEvent(startedJobToken);
    } else if (JobEvent.COMPLETED.equals(event.getJobEventStatus())) {
      try {
        JobStage jobToken = executeNextJob(event);
        if (jobToken == null) {
          JobToken completedJobToken = convertJobTokenJson(event.jobToken());
          jobCompletedEvent(completedJobToken);
        }
      } catch (Exception e) {
        logger.withMessage(
            "Exception occurred while executing next job token in executeNextJobToken()::" + e);
      }
    }
  }

  /**
   * It converts the given jobToken data into JobToken and return it.
   * 
   * @param jobTokenData job token data
   * @return instance of {@link JobToken}
   */
  private JobToken convertJobTokenJson(Object jobTokenData) {
    String jobTokenJson = marshaller.marshall(jobTokenData, DescriptorType.JSON);
    return marshaller.unmarshall(jobTokenJson, JobToken.class, DescriptorType.JSON);
  }

  /**
   * Execute next job stage in the {@link JobPipeline}.
   * 
   * @param jobEvent job event which is received.
   * @throws SchedulerException throws an exception if guid is null.
   */
  private JobStage executeNextJob(JobEvent jobEvent) throws SchedulerException {
    JobToken previousToken = convertJobTokenJson(jobEvent.jobToken());
    String guid = previousToken.getGuid();
    JobStage jobStage = jobPipelineQueue.getNextPipelineJobStage(guid);
    if (jobStage != null) {
      JobPipeline jobPipeline = jobPipelineQueue.getJobPipelineFromGuid(guid);
      if (jobPipeline != null) {
        String serviceName = jobPipeline.getIoConverterServiceName();
        String interfaceName = jobPipeline.getIoConverterInterfaceName();
        JobPipelineIoConverter baseIoConverter =
            (JobPipelineIoConverter) derivedIoConverterFactory.getDerviedInstanceOfConverter(
                JobPipelineIoConverter.class, serviceName, interfaceName);
        Object jobResult = jobEvent.jobResult();
        String jobResultJson = marshaller.marshall(jobResult, DescriptorType.JSON);
        JobToken converterJobToken = jobStageConverter.get().convertJobStageToJobToken(jobStage);
        JobTokenInputInfo jobTokenInputInfo = converterJobToken.getJobTokenInputInfo();
        if (jobTokenInputInfo != null) {
          String ioResultJson = baseIoConverter.convertIoFromTo(jobResultJson,
              jobTokenInputInfo.getInputParameterClassName());
          converterJobToken.addParameter(jobTokenInputInfo.getInputParameterName(), ioResultJson);
        }
        tokenRouterProvider.get().routeJobToken(converterJobToken);
      } else {
        logger.withMessage("JobPipeline getting null.");
      }
    } else {
      logger.withMessage("No JobStages in job stage queue.");
    }
    return jobStage;

  }

  /**
   * Job token completed.
   * 
   * @param token which completed the execution.
   */
  private void jobCompletedEvent(JobToken token) {
    logger.withLevel(Level.INFO).withMessage("Job Execution Completed :" + token.getJobKey());
    // TODO:add record in metrics.
  }

  /**
   * Job token started its execution.
   * 
   * @param token which started the execution.
   */
  private void jobStartedEvent(JobToken token) {
    logger.withLevel(Level.INFO).withMessage("Job Execution started :" + token.getJobKey());
  }

  /**
   * <p>
   * Getter for the field <code>classPathAnalyser</code>.
   * </p>
   *
   * @return a {@link org.appops.core.ClassPathAnalyser} object.
   */
  public ClassPathAnalyser getClassPathAnalyser() {
    return classPathAnalyser;
  }

  /**
   * <p>
   * Setter for the field <code>classPathAnalyser</code>.
   * </p>
   *
   * @param classPathAnalyser a {@link org.appops.core.ClassPathAnalyser} object.
   */
  @Inject
  public void setClassPathAnalyser(ClassPathAnalyser classPathAnalyser) {
    this.classPathAnalyser = classPathAnalyser;
  }

  /**
   * <p>
   * Getter for the field <code>jobPipelineQueue</code>.
   * </p>
   *
   * @return a {@link org.appops.job.JobPipelineQueue} object.
   */
  public JobPipelineQueue getJobPipelineQueue() {
    return jobPipelineQueue;
  }

  /**
   * <p>
   * Setter for the field <code>jobPipelineQueue</code>.
   * </p>
   *
   * @param jobPipelineQueue a {@link org.appops.job.JobPipelineQueue} object.
   */
  @Inject
  public void setJobPipelineQueue(JobPipelineQueue jobPipelineQueue) {
    this.jobPipelineQueue = jobPipelineQueue;
  }

  /**
   * <p>
   * Getter for the field <code>serviceMetaManager</code>.
   * </p>
   *
   * @return a {@link com.google.inject.Provider} object.
   */
  public Provider<ServiceMetaManager> getServiceMetaManager() {
    return serviceMetaManager;
  }

  /**
   * <p>
   * Setter for the field <code>serviceMetaManager</code>.
   * </p>
   *
   * @param serviceMetaManager a {@link com.google.inject.Provider} object.
   */
  @Inject
  public void setServiceMetaManager(Provider<ServiceMetaManager> serviceMetaManager) {
    this.serviceMetaManager = serviceMetaManager;
  }



}
