/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */
package org.appops.scheduler.helper;

import com.google.inject.Inject;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Blob;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.io.IOUtils;
import org.appops.job.JobPipeline;
import org.appops.marshaller.DescriptorType;
import org.appops.marshaller.Marshaller;
import org.appops.scheduler.converter.JobPipelineConverter;
import org.appops.scheduler.dao.JobPipelineDao;
import org.appops.scheduler.pojo.JobPipelinePojo;
import org.hibernate.Hibernate;

public class JobPipelineHelper {
  private JobPipelineDao jobPipelineDao;
  private Logger logger = Logger.getLogger(this.getClass().getSimpleName());
  private JobPipelineConverter jobPipelineConverter;
  @Inject
  private Marshaller marshaller;

  @Inject
  public JobPipelineHelper(JobPipelineDao jobPipelineDao) {
    this.jobPipelineDao = jobPipelineDao;
  }

  /**
   * It saves the job pipeline and return it.
   * 
   * @param jobPipeline instance of {@link JobPipeline}
   * @return saved job pipeline
   */
  public JobPipeline saveJobPipeline(JobPipeline jobPipeline) {
    jobPipelineConverter = new JobPipelineConverter();
    JobPipelinePojo jobPipelinePojo = new JobPipelinePojo();
    jobPipelinePojo = jobPipelineConverter.convertJobPipelineToPojo(jobPipeline);
    jobPipelinePojo.setSchedularExpression(jobPipeline.getScheduleExpression());
    jobPipelinePojo.setJobInfo(getBlob(jobPipeline));
    jobPipelinePojo.setCreatedOn(new Date().toString());
    Integer id = jobPipelineDao.savePojo(jobPipelinePojo);
    if (id != 0) {
      return jobPipeline;
    }
    return null;
  }

  /**
   * It return the instance of {@link Blob} of given job pipeline.
   * 
   * @param jobPipeline instance of {@link JobPipeline}
   * @return instance of {@link Blob}
   */
  private Blob getBlob(JobPipeline jobPipeline) {
    java.sql.Blob data = null;
    String jobPipelineJson = null;
    try {
      jobPipelineJson = marshaller.marshall(jobPipeline, DescriptorType.JSON);
      data = Hibernate.getLobCreator(jobPipelineDao.getPojoSession())
          .createBlob(IOUtils.toByteArray(jobPipelineJson));
    } catch (Exception e) {
      logger.log(Level.SEVERE, "error while converting into blob", e);
    }
    return data;
  }


  /**
   * It returns the list of {@link JobPipeline} of given schedule expression.
   * 
   * @param scheduleAt schedule expression
   * @return list of job pipelines
   * @throws SQLException if occurred in between process
   */
  public List<JobPipeline> getExistingJobPipelines(String scheduleAt) throws SQLException {
    List<JobPipeline> jobPipelinelist = new ArrayList<JobPipeline>();
    List<JobPipelinePojo> jobPipelinePojolist = jobPipelineDao.getExistingJobPipelines(scheduleAt);
    for (JobPipelinePojo pojo : jobPipelinePojolist) {
      JobPipeline jobPipeline = getExistingJobPipelinesFromBlob(pojo.getJobInfo());
      jobPipelinelist.add(jobPipeline);
    }
    return jobPipelinelist;
  }

  private JobPipeline getExistingJobPipelinesFromBlob(Blob jobInfo) throws SQLException {
    String pipelineJson = convertBlobTojson(jobInfo);
    JobPipeline jobPipeline =
        marshaller.unmarshall(pipelineJson, JobPipeline.class, DescriptorType.JSON);
    return jobPipeline;
  }

  private String convertBlobTojson(Blob jobInfo) throws SQLException {
    if (jobInfo == null) {
      return null;
    }
    InputStream is = null;
    try {
      is = jobInfo.getBinaryStream();
      if (is == null) {
        return null;
      }
      byte[] data = new byte[(int) jobInfo.length()];
      if (data.length == 0) {
        return null;
      }
      is.read(data);
      String str = new String(data);
      return str;
    } catch (IOException e) {
      throw new RuntimeException(e);
    } finally {
      if (is != null) {
        try {
          is.close();
        } catch (IOException e) {
          throw new RuntimeException(e);
        }
      }
    }

  }

  /**
   * It returns the job pipeline list.
   * 
   * @return list of {@link JobPipeline}
   */
  public List<JobPipeline> getAllJobPipelines() {
    List<JobPipeline> jobPipelinelist = new ArrayList<JobPipeline>();
    List<JobPipelinePojo> jobPipelinePojolist = jobPipelineDao.getAllJobPipelines();
    for (JobPipelinePojo pojo : jobPipelinePojolist) {
      JobPipeline jobPipeline = null;
      try {
        jobPipeline = getExistingJobPipelinesFromBlob(pojo.getJobInfo());
        jobPipeline.setScheduleExpression(pojo.getSchedularExpression());
      } catch (SQLException e) {
        e.printStackTrace();
      }
      jobPipelinelist.add(jobPipeline);
    }
    return jobPipelinelist;
  }
}
