/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.service.entrypoint;

import java.io.File;
import java.util.HashMap;
import org.appops.core.ServiceException;

/**
 *
 * Service options e.g.
 *
 * java MyService.class --profile.root="./" --service.mode="CLUBBED" --service.root="./"
 *
 * @author deba
 * @version $Id: $Id
 */
public class ServiceArgs {

  /**
   * Identifies folder to look for env-config directory
   */
  // private static String ENV_PROFILE_ROOT = System.getProperty("user.dir");

  /**
   * name of profile / folder within env-config to use for all service configuration information
   * 
   */
  private String profileUsed = "default";
  private String profileRoot = "env-config/";
  private static String CORE_CONFIG = "coreConfig";
  private String deploymentMode = "CLUBBED";

  /**
   * Bind all arguments as top level configuration properties within the injector
   */
  private HashMap<String, String> argMap = new HashMap<String, String>();

  /**
   * <p>
   * Constructor for ServiceArgs.
   * </p>
   *
   * @param args an array of {@link java.lang.String} objects.
   * @throws org.appops.core.ServiceException if any.
   */
  public ServiceArgs(String[] args) throws ServiceException {

    for (String arg : args) {

      if (!arg.startsWith("--")) {

        throw new ServiceException(
            "Invalid arguments while starting service. Service arguments can be provided with format --[argName]=[argValue] \" e.g. java MyServiceEntryPoint --profile.root=\"./\" ) --service.mode=CLUBBED --service.root=./");

      } else {

        String[] keyValue = arg.split("=", 0);
        keyValue[0] = keyValue[0].substring(2);

        if (keyValue.length == 2) {
          argMap.put(keyValue[0], keyValue[1]);
        } else
          System.out.println("Skipped argument without value --> " + keyValue[0]);

      }
    }

    if (argMap.get("profile.name") != null) {
      profileUsed = argMap.get("profile.name");
    }
    if (argMap.get("profile.root") != null) {
      profileRoot = argMap.get("profile.root");
    }
    if (argMap.get("deployment.mode") != null) {
      deploymentMode = argMap.get("deployment.mode");
    }
    addDefaults();
  }


  /**
   * It puts the service config path.
   * 
   * @param serviceConfigPath path of service config file
   * @return instance of {@link ServiceArgs}
   */
  private ServiceArgs withServiceConfig(String serviceConfigPath) {
    if (checkNullOrEmpty(serviceConfigPath)) {
      argMap.put(CORE_CONFIG, serviceConfigPath);
    }
    return this;
  }

  /**
   * It returns the file instance of service configration.
   * 
   * @return service configuration of core.yml file
   */
  public File getServiceConfig() {
    return new File(argMap.get(CORE_CONFIG));
  }

  /**
   * It adds the default path of core.yml file.
   */
  private void addDefaults() {
    withServiceConfig(profileRoot + profileUsed + "/core.yml");
  }

  /**
   * It checks the given value is empty or nor and return approprite result.
   * 
   * @param value to check null or empty
   * @return true, if not null or not empty otherwise false.
   */
  private boolean checkNullOrEmpty(Object value) {
    return (value != null && (value instanceof String ? !((String) value).isEmpty() : true));
  }

  /**
   * <p>
   * getSelectedProfileName.
   * </p>
   *
   * @return a {@link java.lang.String} object.
   */
  public String getSelectedProfileName() {
    return profileUsed;
  }

  /**
   * It returns the profile root.
   * 
   * @return profile root
   */
  public String getProfileRoot() {
    return profileRoot;
  }

  /**
   * It returns the service deployment mode.
   * 
   * @return service deployment mode
   */
  public String getDeploymentMode() {
    return deploymentMode;
  }



  /**
   * <p>
   * getServiceArgs.
   * </p>
   *
   * @return a {@link java.util.HashMap} object.
   */
  public HashMap<String, String> getServiceArgs() {
    return argMap;
  }

}
