/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.service.store;

import java.util.ArrayList;
import java.util.Collection;
import org.appops.core.service.Parameter;
import org.appops.core.service.meta.InterfaceMeta;
import org.appops.core.service.meta.ServiceMeta;
import org.appops.core.service.meta.ServiceOpMeta;

/**
 * The type which stores service definition such as service meta information, service interface
 * definitions,service operation definitions etc.
 *
 */
public interface ServiceMetaRegistry {

  /**
   * Provides service definition instance.
   *
   * @param name Name of service.
   * @return Instance that contains service meta information.
   */
  public ServiceMeta getServiceMeta(String name);

  /**
   * Provides service operation definition.
   *
   * @param pathOrSignature Service operation path or signature.
   * @return Instance that contains service operation meta information.
   */
  public ServiceOpMeta getOpMeta(String pathOrSignature);

  /**
   * Provides service operation definition.
   *
   * @param serviceName name of the service
   * @param pathOrSignature Service operation path or signature.
   * @return Instance that contains service operation meta information.
   */
  public ServiceOpMeta getOpMeta(String serviceName, String pathOrSignature);

  /**
   * Provides service operation definition.
   *
   * @param id Service operation id.
   * @return Instance that contains service operation meta information.
   */
  public ServiceOpMeta getOpMetaById(Long id);

  /**
   * Provides service interface definition.
   *
   * @param service Name of service.
   * @param name Interface name which is to be looked for.
   * @return Instance that contains service interface meta information.
   */
  public InterfaceMeta getInterfaceMeta(String service, String name);

  /**
   * Adds service meta/definition to store.
   *
   * @param serviceMeta Service meta to be added.
   */
  public void addServiceMeta(ServiceMeta serviceMeta);

  /**
   * Adds interface meta to store.
   *
   * @param service Name of service under which interface is to be added.
   * @param intrfcMeta Interface definition which is to be added.
   */
  public void addInterfaceMeta(String service, InterfaceMeta intrfcMeta);

  /**
   * Adds operation meta to store.
   *
   * @param service Name of service under which operation is to be added.
   * @param intrfc Name of interface under which operation is to be added.
   * @param opMeta Operation definition which is to be added.
   */
  public void addOpMeta(String service, String intrfc, ServiceOpMeta opMeta);

  /**
   * <p>
   * getAllServiceMetas.
   * </p>
   *
   * @return a {@link java.util.Collection} object.
   */
  public Collection<ServiceMeta> getAllServiceMetas();

  /**
   * <p>
   * getServicesByPage.
   * </p>
   *
   * @param startIndex a {@link java.lang.Integer} object.
   * @param pageSize a {@link java.lang.Integer} object.
   * @return a {@link java.util.Collection} object.
   */
  public Collection<ServiceMeta> getServicesByPage(Integer startIndex, Integer pageSize);

  /**
   * <p>
   * getInterfaces.
   * </p>
   *
   * @param serviceName a {@link java.lang.String} object.
   * @return a {@link java.util.Collection} object.
   */
  public Collection<InterfaceMeta> getInterfaces(String serviceName);

  /**
   * <p>
   * getOperations.
   * </p>
   *
   * @param serviceName a {@link java.lang.String} object.
   * @param interfaceName a {@link java.lang.String} object.
   * @return a {@link java.util.Collection} object.
   */
  public Collection<ServiceOpMeta> getOperations(String serviceName, String interfaceName);

  /**
   * <p>
   * getParameters.
   * </p>
   *
   * @param serviceName a {@link java.lang.String} object.
   * @param interfaceName a {@link java.lang.String} object.
   * @param operationName a {@link java.lang.String} object.
   * @return a {@link java.util.ArrayList} object.
   */
  public ArrayList<Parameter> getParameters(String serviceName, String interfaceName,
      String operationName);

}
