/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.service.invocation;

import com.google.inject.Inject;
import java.lang.reflect.Method;
import java.util.List;
import org.appops.core.TypeScanner;
import org.appops.core.service.annotation.ServiceOp;
import org.appops.core.service.meta.ServiceOpMeta;
import org.appops.core.service.signature.ServiceOpSignatureBuilder;
import org.appops.service.exception.InvocationException;

/**
 * Locates method from definition provided using reflections.
 *
 * @author deba
 * @version $Id: $Id
 */
public class MethodLocator {

  private TypeScanner scanner;
  private ServiceOpSignatureBuilder signatureBuilder;


  /**
   * Finds method using details passed through operation instance.
   *
   * @param parentClazz Class definition in which method is to be looked for.
   * @param operation Operation instance which contains information to be utilized while locating
   *        method.
   * @return Service method located.
   */
  public Method findMethod(Class<?> parentClazz, ServiceOpMeta operation) {
    String opSignature = getSignatureBuilder().buildOpSignature(operation);
    List<Method> methods = getScanner().getAnnotatedMethods(parentClazz, ServiceOp.class);
    for (Method method : methods) {
      String path = method.getAnnotation(ServiceOp.class).path();
      if (path != null && !path.isEmpty()) {
        if (path.equals(operation.getPath())) {
          return method;
        }
      }
      String methodSignature = getSignatureBuilder().buildMethodSignature(parentClazz, method);

      if (methodSignature != null && methodSignature.equals(opSignature)) {
        return method;
      }
    }
    throw new InvocationException("Unable to locate method -> " + operation.getName() + ", in "
        + parentClazz.getCanonicalName());
  }

  /**
   * <p>
   * Getter for the field <code>scanner</code>.
   * </p>
   *
   * @return a {@link org.appops.core.TypeScanner} object.
   */
  public TypeScanner getScanner() {
    return scanner;
  }

  /**
   * <p>
   * Setter for the field <code>scanner</code>.
   * </p>
   *
   * @param scanner a {@link org.appops.core.TypeScanner} object.
   */
  @Inject
  public void setScanner(TypeScanner scanner) {
    this.scanner = scanner;
  }

  /**
   * <p>
   * Getter for the field <code>signatureBuilder</code>.
   * </p>
   *
   * @return a {@link org.appops.core.service.signature.ServiceOpSignatureBuilder} object.
   */
  public ServiceOpSignatureBuilder getSignatureBuilder() {
    return signatureBuilder;
  }

  /**
   * <p>
   * Setter for the field <code>signatureBuilder</code>.
   * </p>
   *
   * @param signatureBuilder a {@link org.appops.core.service.signature.ServiceOpSignatureBuilder}
   *        object.
   */
  @Inject
  public void setSignatureBuilder(ServiceOpSignatureBuilder signatureBuilder) {
    this.signatureBuilder = signatureBuilder;
  }

}
