/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.service.invocation;

import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.Provider;
import com.google.inject.servlet.RequestParameters;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import org.appops.core.service.OpParameterMap;
import org.appops.core.service.Parameter;
import org.appops.core.service.meta.MappedOpMeta;
import org.appops.core.service.meta.ServiceOpMeta;
import org.appops.logging.impl.LogManager;
import org.appops.logging.logger.Logger;
import org.appops.logging.meta.Level;
import org.appops.marshaller.DescriptorType;
import org.appops.marshaller.Marshaller;
import org.appops.service.exception.InvocationException;
import org.appops.service.lifecycle.CallStack;
import org.appops.service.store.RequestParameterStore;
import org.appops.slim.base.api.ServiceMetaManager;
import org.appops.slim.base.invocation.UrlEncodeUtil;

/**
 * Invoker class which locates and invokers mapped code based method for an operation definition
 * provided.
 *
 * @author deba
 * @version $Id: $Id
 */
public class OperationInvoker {

  private Provider<CallStack> callStackProvider;
  private Provider<MethodLocator> methodLocator;
  private Provider<ServiceMetaManager> serviceMetaManager;
  private Marshaller marshaller;
  private Provider<RequestParameterStore> paramStoreProvider;
  private static final String LOG_SERVICE_NAME = "LogViewer";
  private Logger rootLogger;

  @Inject
  private Injector injector;

  @Inject
  @RequestParameters
  private Provider<Map<String, String[]>> requestParametersProvider;

  @Inject
  /**
   * <p>
   * Constructor for OperationInvoker.
   * </p>
   *
   * @param serviceMetaManager a {@link com.google.inject.Provider} object.
   * @param logManager a {@link org.appops.logging.impl.LogManager} object.
   */
  public OperationInvoker(Provider<ServiceMetaManager> serviceMetaManager, LogManager logManager) {
    this.serviceMetaManager = serviceMetaManager;
    this.rootLogger = logManager.getRootLogger();
  }

  /**
   * Locates and invokes method for an operation definition passed.
   *
   * @param service Name of service.
   * @param opPathOrSignature Operation path. Can be signature or a full path to reach the method.
   * @return Result obtained from actual method invocation.
   */
  public Object invoke(String service, String opPathOrSignature) {
    return invoke(service, opPathOrSignature, null);
  }

  /**
   * Locates and invokes method for an operation definition passed.
   *
   * @param serviceName Name of the service.
   * @param opPathOrSignature Operation path. Can be signature or a full path to reach the method.
   * @param postData Operation parameters received as post request data.
   * @return Result obtained from actual method invocation.
   */
  public Object invoke(String serviceName, String opPathOrSignature, OpParameterMap postData) {
    ServiceOpMeta opMeta = getServiceStore().getOpMeta(serviceName, opPathOrSignature);
    ServiceOpMeta finalOperation = getMappedOperation(opMeta);

    printOpLog(Level.INFO, serviceName, finalOperation.getName(), "Starting operation execution.");
    String apiQualifiedName = finalOperation.getParent().getQualifiedClassName();
    Class<?> apiClass;
    List<Object> params = getParameters(postData, opPathOrSignature, finalOperation);
    this.paramStoreProvider.get().setParameters(params);
    try {
      apiClass = Class.forName(apiQualifiedName);
    } catch (ClassNotFoundException e1) {
      printOpLog(Level.ERROR, serviceName, finalOperation.getName(), "Api/Operation  not found!!!");
      throw new InvocationException("Api  not found, name ->" + apiQualifiedName);
    }

    Method method = getMethodLocator().findMethod(apiClass, finalOperation);
    Object[] paramRaw = params.toArray();

    Object result = null;
    try {
      Object service = injector.getInstance(apiClass);
      if (method.getParameterTypes().length > 0 && method.isVarArgs()) {
        Object argumentArray = new Object[] {paramRaw};
        result = method.invoke(service, argumentArray);
      } else {
        result = method.invoke(service, paramRaw);
      }
    } catch (Exception e) {
      throw new InvocationException(e.getCause());
    }
    printOpLog(Level.INFO, serviceName, finalOperation.getName(), "Completed operation execution.");
    return result;
  }

  /**
   * Friendly method to generate logs during operation execution.
   * 
   * @param level Log level. eg. Level.INFO, Level.ERROR etc.
   * @param service String service name.
   * @param opName Name of the operation under execution.
   * @param logMessage Additional log message.
   */
  private void printOpLog(Level level, String service, String opName, String logMessage) {
    if (LOG_SERVICE_NAME.toLowerCase().equals(service.toLowerCase())) {
      return;
    }
    String finalMessage =
        "Service operation log :  Service_Name -> " + service + ", Operation_Name -> " + opName;
    if (logMessage != null && !logMessage.isEmpty()) {
      finalMessage += ", Detailed message :- " + logMessage;
    }
    rootLogger.withLevel(level).withMessage(finalMessage).log();

  }

  private List<Object> getParameters(OpParameterMap postData, String opPathOrSignature,
      ServiceOpMeta finalOperation) {
    List<Object> params;
    if (postData != null && !postData.isEmpty()) {
      params = fromPostData(postData, finalOperation);
    } else {
      params =
          fromRequestOrPathParameters(opPathOrSignature, finalOperation, getRequestParameters());
    }
    return params;
  }

  private List<Object> fromPostData(OpParameterMap postData, ServiceOpMeta finalOperation) {
    List<Object> paramsToReturn = new ArrayList<>();
    for (Integer order : postData.keySet()) {

      Parameter parameter = postData.get(order);
      Object value = parameter.getValue();
      parameter = finalOperation.getParameters().get(order);
      Class<?> typeClazz;
      try {
        if (parameter.getTypeName().equals("java.lang.Byte[]")) {
          parameter.setTypeName("[Ljava.lang.Byte;");
        }
        typeClazz = Class.forName(parameter.getTypeName());
      } catch (Exception e) {
        throw new InvocationException("Cannot find type for parameter -> " + parameter.getName()
            + " type-name -> " + parameter.getTypeName());
      }
      if (!typeClazz.isAssignableFrom(value.getClass())) {
        String stringVal;
        if (value instanceof String) {
          stringVal = (String) value;
          if (typeClazz.isEnum()) {
            value = Enum.valueOf((Class<Enum>) typeClazz, stringVal);
            stringVal = null;
          }
        } else {
          stringVal = getMarshaller().marshall(value, DescriptorType.JSON);
        }
        if (stringVal != null) {
          value = getMarshaller().unmarshall(stringVal, typeClazz, DescriptorType.JSON);
        }
      }

      paramsToReturn.add(value);
    }
    return paramsToReturn;
  }



  /**
   * Converts method parameters to actual types defined in op meta.
   * 
   * @param path Path of service operation.
   * @param finalOperation Operation meta information.
   * @param requestParameters Parameters fetched from current request.
   * 
   * @return List of typed method parameters.
   */
  private List<Object> fromRequestOrPathParameters(String path, ServiceOpMeta finalOperation,
      Map<String, String[]> requestParameters) {
    List<String[]> requestParamList = new LinkedList<>(requestParameters.values());
    Map<String, String> pathParams = extractPathParameters(path, finalOperation.getPath());
    List<Object> paramList = new ArrayList<>();
    for (Parameter parameter : finalOperation.getParameters().values()) {
      String parameterName = parameter.getName();
      String strVal = null;
      if (pathParams.containsKey(parameterName)) {
        strVal = pathParams.get(parameterName);
      } else {
        String[] valArr = null;
        if (requestParameters.containsKey(parameterName)) {
          valArr = requestParameters.get(parameterName);
        } else {
          int order = parameter.getOrder();
          valArr = requestParamList.get(order);
        }
        if (valArr != null && valArr.length > 0) {
          strVal = valArr[0];
        }
      }

      Class<?> typeClazz;
      try {
        typeClazz = Class.forName(parameter.getTypeName());
      } catch (Exception e) {
        throw new InvocationException("Cannot find type for parameter -> " + parameter.getName()
            + " type-name -> " + parameter.getTypeName());
      }
      if (strVal != null && !strVal.equals("null")) {
        if (String.class.isAssignableFrom(typeClazz)) {
          paramList.add(strVal);
        } else {
          Object value = getMarshaller().unmarshall(strVal, typeClazz, DescriptorType.JSON);
          paramList.add(value);
        }
      } else {
        paramList.add(null);
      }
    }
    return paramList;
  }


  /**
   * Extracts parameters from meta and actual paths passed.
   * 
   * @param path Actual operation path to be used for invocation. e.g. users/addUser/1/vedang
   * @param metaPath Path definition. e.g. users/addUser/{id}/{name}
   */
  private Map<String, String> extractPathParameters(String path, String metaPath) {
    UrlEncodeUtil decoder = new UrlEncodeUtil();
    Map<String, String> pathParams = new HashMap<>();
    if (metaPath != null && metaPath.contains("/") && path.contains("/")) {
      String[] metaElements = metaPath.split("/");
      String[] pathElements = path.split("/");
      for (int i = 0; i < metaElements.length; i++) {
        String metaElement = metaElements[i];
        if (metaElement.startsWith("{") && metaElement.endsWith("}")) {
          String parameterName =
              metaElement.substring(metaElement.indexOf("{") + 1, metaElement.indexOf("}")).trim();
          String value = pathElements[i];
          pathParams.put(parameterName, decoder.decodePathParam(value));
        }
      }
    }
    return pathParams;
  }



  /**
   * it gets the mapped operation.
   * 
   * @param op indicates the operation.
   * @return operation.
   */
  private ServiceOpMeta getMappedOperation(ServiceOpMeta op) {

    if (op instanceof MappedOpMeta) {
      ServiceOpMeta mappedDestination =
          getServiceStore().getOpMeta(((MappedOpMeta) op).getMappedTo());
      if (mappedDestination.isDynamic()) {
        getCallStack().addOperation(mappedDestination);
      }
      if (mappedDestination instanceof MappedOpMeta) {
        return getMappedOperation(mappedDestination);
      }
      return mappedDestination;
    } else {
      return op;
    }

  }


  /**
   * <p>
   * getCallStack.
   * </p>
   *
   * @return a {@link org.appops.service.lifecycle.CallStack} object.
   */
  public CallStack getCallStack() {
    return callStackProvider.get();
  }

  /**
   * <p>
   * Setter for the field <code>callStackProvider</code>.
   * </p>
   *
   * @param callStackProvider a {@link com.google.inject.Provider} object.
   */
  @Inject
  public void setCallStackProvider(Provider<CallStack> callStackProvider) {
    this.callStackProvider = callStackProvider;
  }



  private Map<String, String[]> getRequestParameters() {
    return requestParametersProvider.get();
  }



  /**
   * <p>
   * Getter for the field <code>methodLocator</code>.
   * </p>
   *
   * @return a {@link org.appops.service.invocation.MethodLocator} object.
   */
  public MethodLocator getMethodLocator() {
    return methodLocator.get();
  }

  /**
   * <p>
   * Setter for the field <code>methodLocator</code>.
   * </p>
   *
   * @param methodLocator a {@link com.google.inject.Provider} object.
   */
  @Inject
  public void setMethodLocator(Provider<MethodLocator> methodLocator) {
    this.methodLocator = methodLocator;
  }



  /**
   * <p>
   * Setter for the field <code>requestParametersProvider</code>.
   * </p>
   *
   * @param requestParametersProvider a {@link com.google.inject.Provider} object.
   */
  public void setRequestParametersProvider(
      Provider<Map<String, String[]>> requestParametersProvider) {
    this.requestParametersProvider = requestParametersProvider;
  }



  /**
   * <p>
   * getServiceStore.
   * </p>
   *
   * @return a {@link org.appops.slim.base.api.ServiceMetaManager} object.
   */
  public ServiceMetaManager getServiceStore() {
    return serviceMetaManager.get();
  }

  /**
   * <p>
   * Getter for the field <code>marshaller</code>.
   * </p>
   *
   * @return a {@link org.appops.marshaller.Marshaller} object.
   */
  public Marshaller getMarshaller() {
    return marshaller;
  }

  /**
   * <p>
   * Setter for the field <code>marshaller</code>.
   * </p>
   *
   * @param marshaller a {@link org.appops.marshaller.Marshaller} object.
   */
  @Inject
  public void setMarshaller(Marshaller marshaller) {
    this.marshaller = marshaller;
  }

  /**
   * <p>
   * Getter for the field <code>paramStoreProvider</code>.
   * </p>
   *
   * @return a {@link com.google.inject.Provider} object.
   */
  public Provider<RequestParameterStore> getParamStoreProvider() {
    return paramStoreProvider;
  }

  /**
   * <p>
   * Setter for the field <code>paramStoreProvider</code>.
   * </p>
   *
   * @param paramStoreProvider a {@link com.google.inject.Provider} object.
   */
  @Inject
  public void setParamStoreProvider(Provider<RequestParameterStore> paramStoreProvider) {
    this.paramStoreProvider = paramStoreProvider;
  }

  /**
   * <p>
   * Setter for the field <code>injector</code>.
   * </p>
   *
   * @param injector a {@link com.google.inject.Injector} object.
   */
  public void setInjector(Injector injector) {
    this.injector = injector;
  }

  /**
   * <p>
   * Getter for the field <code>serviceMetaManager</code>.
   * </p>
   *
   * @return a {@link com.google.inject.Provider} object.
   */
  public Provider<ServiceMetaManager> getServiceMetaManager() {
    return serviceMetaManager;
  }

  /**
   * <p>
   * Setter for the field <code>serviceMetaManager</code>.
   * </p>
   *
   * @param serviceMetaManager a {@link com.google.inject.Provider} object.
   */
  public void setServiceMetaManager(Provider<ServiceMetaManager> serviceMetaManager) {
    this.serviceMetaManager = serviceMetaManager;
  }



}
