/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.service.request;

import com.google.inject.Inject;
import com.google.inject.Provider;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.Part;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.appops.core.mime.MimeType;
import org.appops.core.service.OpParameterMap;
import org.appops.core.service.RequestMethod;
import org.appops.core.service.meta.ServiceMeta;
import org.appops.logging.logger.Logger;
import org.appops.logging.meta.Level;
import org.appops.marshaller.DescriptorType;
import org.appops.marshaller.Marshaller;
import org.appops.service.exception.InvocationException;
import org.appops.service.invocation.OperationInvoker;
import org.appops.slim.base.api.ServiceMetaManager;

public class ServiceApiFilter implements Filter {

  @Inject
  private Provider<OperationInvoker> invoker;
  @Inject
  private Provider<RestResponseWriter> responseWriter;
  @Inject
  private Logger logger;
  @Inject
  private Marshaller marshaller;

  private static final String MULTIPART_CONFIG_ELEMENT = "org.eclipse.jetty.multipartConfig";
  static final String RESPONSE_TYPE_PARAMETER = "RESPONSE_TYPE";
  static final String HEAD_REQUEST = "HEAD_REQUEST";
  private File file = null;
  private ArrayList<File> filesList = null;

  @Inject
  private ServiceMetaManager serviceMetaManager;

  @Override
  public void init(FilterConfig filterConfig) throws ServletException {

  }

  @Override
  public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
      throws IOException, ServletException {

    HttpServletRequest req = (HttpServletRequest) request;
    HttpServletResponse resp = (HttpServletResponse) response;

    String reqUri = req.getRequestURI();
    if (reqUri.contains("/Op/")) {
      reqUri = reqUri.replaceAll("/Op/", "/");
    }
    String serviceName = getServiceNameFromReqUri(reqUri);
    ServiceMeta serviceMeta = null;
    if (serviceName != null && !serviceName.isEmpty()) {
      try {
        serviceMeta = serviceMetaManager.getServiceMeta(serviceName);
      } catch (Exception e) {

      }
    }

    if (serviceMeta != null) {

      String requestedMethod = req.getMethod();
      if (RequestMethod.GET.name().equals(requestedMethod)) {
        doGetRequest(req, resp, reqUri);
      } else if (RequestMethod.POST.name().equals(requestedMethod)) {
        doPostRequest(req, resp, reqUri);
      }
    } else {
      chain.doFilter(request, response);
    }

  }

  public void doGetRequest(HttpServletRequest req, HttpServletResponse resp, String reqUri) {

    try {
      String[] serviceAndPathArr = extractServiceAndOpPath(reqUri);
      Object result = invoker.get().invoke(serviceAndPathArr[0], serviceAndPathArr[1]);
      if (result != null) {
        responseWriter.get().sendResponse(resp, (Object) result,
            (String) req.getParameter(RESPONSE_TYPE_PARAMETER));
      }
    } catch (Exception e) {
      String msg = e.getMessage() != null ? e.getMessage() : Arrays.toString(e.getStackTrace());
      sendErronInResponse(resp, HttpServletResponse.SC_INTERNAL_SERVER_ERROR, msg);
    }
  }

  public void doPostRequest(HttpServletRequest req, HttpServletResponse resp, String reqUri) {
    boolean hasFormData = req.getContentType() != null && !req.getContentType().isEmpty()
        && req.getContentType().toLowerCase().contains(MimeType.MULTIPART.value().toLowerCase());
    try {
      String[] serviceAndPathArr = extractServiceAndOpPath(reqUri);
      OpParameterMap parameterMap = null;
      if (hasFormData) {
        parameterMap = convertFormDataToParamMap(req.getParts());
      } else {
        parameterMap = unmarshallParameterMap(req);
      }
      Object result =
          invoker.get().invoke(serviceAndPathArr[0], serviceAndPathArr[1], parameterMap);
      if (result != null) {
        responseWriter.get().sendResponse(resp, result,
            (String) req.getParameter(RESPONSE_TYPE_PARAMETER));
      }
      if (filesList != null) {
        for (File file : filesList) {
          file.delete();
        }
      }
    } catch (Exception e) {
      sendErronInResponse(resp, HttpServletResponse.SC_INTERNAL_SERVER_ERROR, e.getMessage());
    }

  }

  /**
   * Returns the service name from url.
   * 
   * @param reqUri request URL
   * @return service name
   */
  private String getServiceNameFromReqUri(String reqUri) {
    if (reqUri.startsWith("/")) {
      reqUri = reqUri.substring(1);
    }

    if (reqUri.contains(".html")) {
      int indexOfHtml = reqUri.indexOf(".html");
      reqUri = reqUri.substring(indexOfHtml + 5);

    } else if (reqUri.contains(".jsp")) {
      int indexOfJsp = reqUri.indexOf(".jsp");
      reqUri = reqUri.substring(indexOfJsp + 4);
    }

    if (reqUri.startsWith("/")) {
      reqUri = reqUri.substring(1);
    }

    String[] requestPathArr = reqUri.split("/");
    return requestPathArr[0];
  }

  /**
   * Sends appropriate error code and message in response if anything goes wrong while invoking
   * service operation.
   *
   * @param resp Servlet response instance.
   * @param errorCode Integer value representing type of error e.g. 500 for
   *        HttpServletResponse.SC_INTERNAL_SERVER_ERROR
   * @param message Simple message to explain cause of failure.
   */
  public void sendErronInResponse(HttpServletResponse resp, int errorCode, String message) {
    try {
      if (message.contains("Invalid request :")) {
        errorCode = HttpServletResponse.SC_BAD_REQUEST;
      }
      resp.sendError(errorCode, message);
    } catch (Exception ex) {
      logger.withLevel(Level.ERROR).withMessage(ex.toString()).log();
    }
  }

  /**
   * Extracts service name and operation path from request path provided.
   * 
   * @param requestPath Servlet path received from request.
   * 
   * @return array of service name and operation path to reach service .
   */
  private String[] extractServiceAndOpPath(String requestPath) {
    if (requestPath.startsWith("/")) {
      requestPath = requestPath.substring(1);
    }

    if (requestPath.isEmpty()) {
      throwRequestPatternException();
    }

    String[] requestPathArr = requestPath.split("/", 2);

    return requestPathArr;
  }

  /**
   * It throws the request pattern exception.
   */
  public void throwRequestPatternException() {
    throw new InvocationException(
        "Invalid request : minimum required pattern is " + "/Op/<service_name>/<operation_info>"
            + " An operation_info can be name or path of an operation.");
  }

  void setResponseWriter(Provider<RestResponseWriter> responseWriter) {
    this.responseWriter = responseWriter;
  }

  void setInvoker(Provider<OperationInvoker> invoker) {
    this.invoker = invoker;
  }

  void setLogger(Logger logger) {
    this.logger = logger;
  }


  /**
   * Converts a request data to {@link OpParameterMap}.
   * 
   * @param parts Part which contains data.
   * @return Instance of {@link OpParameterMap}.
   */
  private OpParameterMap convertFormDataToParamMap(Collection<Part> parts) throws IOException {
    OpParameterMap parameterMap = new OpParameterMap();
    int order = 0;
    filesList = new ArrayList<File>();
    for (Part part : parts) {
      Object value = null;
      if (part.getSubmittedFileName() != null) {
        value = convertPartToFile(part);
        filesList.add((File) value);
      } else {
        value = IOUtils.toString(part.getInputStream(), StandardCharsets.UTF_8.name());
      }
      parameterMap.addParameter(order++, part.getName(), value);
    }
    return parameterMap;
  }

  /**
   * Converts a Part to {@link java.io.File}.
   *
   * @param part Part which contains instance of {@link java.io.File}.
   * @return Instance of {@link java.io.File}.
   */
  public File convertPartToFile(Part part) {
    try {
      final String path =
          Paths.get(".").toAbsolutePath().normalize().toString() + "/src/main/resources/";
      final String fileName = part.getName();
      file = new File(path + fileName);
      FileUtils.copyInputStreamToFile(part.getInputStream(), file);
      return file;
    } catch (Exception e) {
      throw new InvocationException(e.getMessage());
    }
  }

  /**
   * Converts a request data to {@link OpParameterMap}.
   * 
   * @param request Instance of {@link HttpServletRequest}.
   * @return Instance of {@link OpParameterMap}.
   */
  private OpParameterMap unmarshallParameterMap(HttpServletRequest request) {
    StringBuilder jb = new StringBuilder();
    String line = null;
    try {
      BufferedReader reader = request.getReader();
      while ((line = reader.readLine()) != null) {
        jb.append(line);
      }
    } catch (Exception e) {
      throw new InvocationException("Unable to read request data");
    }
    String data = jb.toString();
    if (data == null || data.isEmpty()) {
      return null;
    }
    try {
      OpParameterMap parameterMap =
          marshaller.unmarshall(data, OpParameterMap.class, DescriptorType.JSON);
      return parameterMap;
    } catch (Exception e) {
      throw new InvocationException("Unable to unmarshall post request data.", e.getCause());
    }

  }



  @Override
  public void destroy() {

  }


}
