/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.service.invocation;

import com.google.inject.Inject;
import com.google.inject.Provider;
import java.lang.reflect.Method;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.appops.core.service.annotation.ServiceOp;
import org.appops.core.service.meta.ServiceOpMeta;
import org.appops.core.service.signature.ServiceOpSignatureBuilder;
import org.appops.service.exception.InvocationException;
import org.appops.service.lifecycle.CallStack;
import org.appops.slim.base.api.ServiceMetaManager;

/**
 * <p>
 * OperationDecorator class.
 * </p>
 *
 * @author deba
 * @version $Id: $Id
 */
public class OperationDecorator implements MethodInterceptor {
  private Provider<CallStack> callStackProvider;
  private ServiceOpSignatureBuilder signatureBuilder;
  private Provider<ServiceMetaManager> serviceMetaManager;



  /** {@inheritDoc} */
  @Override
  public Object invoke(MethodInvocation mi) throws Throwable {
    Method method = mi.getMethod();

    ServiceOpMeta operation = findOperation(method);

    getCallStack().addOperation(operation);
    Object res = null;
    try {
      res = mi.proceed();
    } catch (Exception e) {
      getCallStack().failedWithException(e);
      throw new InvocationException("Error while executing jobs for operation ::"
          + operation.getName() + ":: " + e.getMessage());
    } finally {
      getCallStack().callFinished();
    }
    return res;
  }

  private ServiceOpMeta findOperation(Method method) {

    if (method.isAnnotationPresent(ServiceOp.class)) {
      String signature = getSignatureBuilder().buildMethodSignature(method.getClass(), method);
      return getServiceStore().getOpMeta(signature);
    }

    throw new InvocationException("Operation not found for ->" + method.getName());
  }



  /**
   * <p>
   * getCallStack.
   * </p>
   *
   * @return a {@link org.appops.service.lifecycle.CallStack} object.
   */
  public CallStack getCallStack() {
    return callStackProvider.get();

  }


  /**
   * <p>
   * Setter for the field <code>callStackProvider</code>.
   * </p>
   *
   * @param callStackProvider a {@link com.google.inject.Provider} object.
   */
  @Inject
  public void setCallStackProvider(Provider<CallStack> callStackProvider) {
    this.callStackProvider = callStackProvider;
  }

  /**
   * <p>
   * setServiceStore.
   * </p>
   *
   * @param serviceMetaManager a {@link com.google.inject.Provider} object.
   */
  @Inject
  public void setServiceStore(Provider<ServiceMetaManager> serviceMetaManager) {
    this.serviceMetaManager = serviceMetaManager;
  }

  /**
   * <p>
   * Getter for the field <code>signatureBuilder</code>.
   * </p>
   *
   * @return a {@link org.appops.core.service.signature.ServiceOpSignatureBuilder} object.
   */
  public ServiceOpSignatureBuilder getSignatureBuilder() {
    return signatureBuilder;
  }

  /**
   * <p>
   * Setter for the field <code>signatureBuilder</code>.
   * </p>
   *
   * @param signatureBuilder a {@link org.appops.core.service.signature.ServiceOpSignatureBuilder}
   *        object.
   */
  @Inject
  public void setSignatureBuilder(ServiceOpSignatureBuilder signatureBuilder) {
    this.signatureBuilder = signatureBuilder;
  }


  /**
   * <p>
   * getServiceStore.
   * </p>
   *
   * @return a {@link org.appops.slim.base.api.ServiceMetaManager} object.
   */
  public ServiceMetaManager getServiceStore() {
    return serviceMetaManager.get();
  }
}
