/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.service.lifecycle;

import java.util.EmptyStackException;
import java.util.NoSuchElementException;
import java.util.Stack;

import org.appops.core.service.meta.ServiceMeta;
import org.appops.core.service.meta.ServiceOpMeta;
import org.appops.service.exception.InvocationException;
import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.Provider;

/**
 * A stack of operation invocation. It stores operations in the order in which they are invoked.
 *
 * @author deba
 * @version $Id: $Id
 */
public class CallStack extends Stack<Call> {

  private Exception exception;


  /**
   * Adds an invoked operation into call stack.
   *
   * @param operation Operation instance which is invoked.
   */
  public void addOperation(final ServiceOpMeta operation) {
    Call current = null;
    if (!isEmpty()) {
      current = getCurrentCall();
    }

    final Call newCall = new Call(operation, current);
    push(newCall);
  }

  /**
   * The first service in the chain. Typically its the service that user is accessing.
   * 
   * @return service .
   */
  public ServiceMeta getServiceOfOrigin() {
    return getCallOfOrigin().getOperation().serviceMeta();
  }

  /**
   * The first service in the chain. Typically its the service that user is accessing.
   * 
   * @return service name .
   */
  public String getServiceOfOriginName() {
    return getCallOfOrigin().getOperation().serviceName();
  }

  /**
   * @return current service object for the call in progress
   */
  public ServiceMeta getCurrentService() {
    return getCurrentOperation().serviceMeta();
  }

  /**
   * @return current service name for the call in progress
   */
  public String getCurrentServiceName() {
    return getCurrentOperation().serviceName();
  }

  /**
   * Removes finished call from stack.
   */
  public void callFinished() {
    try {
      pop();
    } catch (final EmptyStackException ex) {
      throw new InvocationException(ex);
    }
  }

  /**
   * Fetches the caller operation details.
   *
   * @return The caller call object of the current service operation. Service object contained Must
   *         be different from the service whose operation is being executed . Returns null if its
   *         the first call in chain
   */
  protected Call getCaller() {
    if (getCurrentCall() == null) {
      throw new InvocationException("Caller not present in stack.");
    }
    return getCurrentCall().getParent();
  }

  /**
   * Fetches the caller operation.
   *
   * @return Operation that called the current one
   */
  public ServiceOpMeta getCallerOperation() {
    final Call caller = getCaller();

    if (caller != null) {
      return caller.getOperation();
    } else {
      throw new InvocationException("Caller not present in stack.");
    }
  }

  /**
   * Fetches the origin of invocation chain.
   *
   * @return First call object in chain
   */
  protected Call getCallOfOrigin() {
    Call serv = null;
    try {
      serv = this.firstElement();
    } catch (final NoSuchElementException ex) {
      throw new InvocationException(ex);
    }
    return serv;
  }

  /**
   * Fetches current operation call information.
   *
   * @return Current call object for current service operation in execution
   */
  public Call getCurrentCall() {
    if (!isEmpty()) {
      return peek();
    }
    throw new InvocationException("Cannot fetch current call, stack is empty.");
  }

  /**
   * Fetches current operation in invocation.
   *
   * @return current service operation in execution
   */
  public ServiceOpMeta getCurrentOperation() {
    return getCurrentCall().getOperation();
  }

  /**
   * Fetches origin of invocation chain.
   *
   * @return first operation in call chain
   */
  public ServiceOpMeta getOperationOfOrigin() {
    return getCallOfOrigin().getOperation();
  }

  /**
   * <p>
   * failedWithException.
   * </p>
   *
   * @param e a {@link java.lang.Exception} object.
   */
  public void failedWithException(Exception e) {
    this.exception = e;
  }

  /**
   * <p>
   * getFailureCause.
   * </p>
   *
   * @return a {@link java.lang.Exception} object.
   */
  public Exception getFailureCause() {
    return exception;
  }


}
