/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */
package org.appops.service;

import com.google.inject.Inject;
import com.google.inject.Provider;
import java.lang.annotation.Annotation;
import org.appops.core.deployment.ServiceConfiguration;
import org.appops.core.service.ExecutionMode;
import org.appops.core.service.meta.ServiceMeta;
import org.appops.service.exception.DeploymentException;
import org.appops.service.generator.ServiceMetaGenerator;
import org.appops.slim.base.api.ServiceMetaManager;

/**
 * <p>
 * Abstract ServiceInitializer class.
 * </p>
 *
 * @author deba
 * @version $Id: $Id
 */
public abstract class ServiceInitializer {

  private Provider<ServiceMetaGenerator> serviceMetaGenerator;
  private Provider<ServiceMetaManager> serviceMetaManager;
  @Inject
  private CurrentDeploymentProvider currentDeploymentProvider;


  @Inject
  /**
   * <p>
   * Constructor for ServiceInitializer.
   * </p>
   *
   * @param serviceMetaGenerator a {@link com.google.inject.Provider} object.
   * @param serviceMetaManager a {@link com.google.inject.Provider} object.
   */
  public ServiceInitializer(Provider<ServiceMetaGenerator> serviceMetaGenerator,
      Provider<ServiceMetaManager> serviceMetaManager) {
    this.serviceMetaGenerator = serviceMetaGenerator;
    this.serviceMetaManager = serviceMetaManager;
  }


  /**
   * <p>
   * initialize.
   * </p>
   *
   * @param serviceName a {@link java.lang.String} object.
   * @param config a {@link ServiceConfiguration} object.
   * @param serviceAnnotation a {@link java.lang.Class} object.
   * @return a {@link java.lang.String} object.
   */
  public abstract String initialize(String serviceName, ServiceConfiguration config,
      Class<? extends Annotation> serviceAnnotation);

  /**
   * Scans and adds service meta definition to service store.
   *
   * @param serviceName service name
   * @param config instance of {@link ServiceConfiguration}
   * @param serviceAnnotation service annotation
   * @return Name of service.
   */
  protected String addServiceMeta(String serviceName, ServiceConfiguration config,
      Class<? extends Annotation> serviceAnnotation) {
    try {
      ServiceMeta serviceMeta;
      if (config.getExecutionMode().equals(ExecutionMode.JOB)) {
        serviceMeta = serviceMetaGenerator.get().generateJesMeta(serviceName,
            config.getWebConfig().getContextPath(), config.serviceUrl(), "", serviceAnnotation);
      } else {
        serviceMeta = serviceMetaGenerator.get().generateServiceMeta(serviceName,
            config.getWebConfig().getContextPath(), config.serviceUrl(), serviceAnnotation,
            config.getPackageToScan());
      }
      currentDeploymentProvider.get().setGatewayUrl(config.getGatewayUrl());
      serviceMetaManager.get().registerServiceMeta(serviceMeta);
      return serviceMeta.getName();
    } catch (Exception e) {
      throw new DeploymentException("Failed to add service meta :: " + e);
    }
  }

}
