/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.service.generator;

import com.google.inject.Inject;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.Set;
import org.appops.cache.event.EventSubscriber;
import org.appops.cache.event.Subscriber;
import org.appops.cache.slim.ServiceBus;
import org.appops.core.ClassPathAnalyser;
import org.appops.core.TypeScanner;
import org.appops.core.annotation.Job;
import org.appops.core.annotation.SubscriberListener;
import org.appops.core.job.JobMeta;
import org.appops.core.service.ExecutionMode;
import org.appops.core.service.ServiceRoute;
import org.appops.core.service.annotation.ServiceOp;
import org.appops.core.service.meta.InterfaceMeta;
import org.appops.core.service.meta.ServiceMeta;
import org.appops.core.service.meta.ServiceOpMeta;
import org.appops.service.job.JobExecService;
import org.appops.service.store.JobMetaRegistry;

/**
 * Generates service meta definition by scanning artifact jar.
 *
 * @author deba
 * @version $Id: $Id
 */
public class ServiceMetaGenerator {

  private ClassPathAnalyser classPathAnalyser;
  private JobMetaRegistry jobMetaRegistry;
  private ServiceBus serviceBus;

  /**
   * Generates service meta from service jar provided.
   *
   * @param serviceName Name of the service.
   * @param routeContext Route context for service app.
   * @param serviceUrl Qualified url for service app.
   * @param serviceAnnotation Service annotation.
   * @return Service meta information generated from service jar scanning.
   */
  public ServiceMeta generateServiceMeta(String serviceName, String routeContext, String serviceUrl,
      Class<? extends Annotation> serviceAnnotation) {
    return generateServiceMeta(serviceName, routeContext, serviceUrl, serviceAnnotation, "");

  }

  /**
   * Generates service meta from service jar provided.
   *
   * @param serviceName Name of the service.
   * @param routeContext Route context for service app.
   * @param serviceUrl Qualified url for service app.
   * @param serviceAnnotation Service annotation.
   * @param refPackage Project base package.
   * @return Service meta information generated from service jar scanning.
   */
  public ServiceMeta generateServiceMeta(String serviceName, String routeContext, String serviceUrl,
      Class<? extends Annotation> serviceAnnotation, String refPackage) {
    ServiceMeta serviceMeta = new ServiceMeta(serviceName);
    serviceMeta.setRoute(new ServiceRoute(routeContext, serviceUrl));
    addServiceDetails(serviceAnnotation, serviceMeta, refPackage);
    addSubscriberDetails(serviceMeta, refPackage);
    return serviceMeta;

  }

  /**
   * Generates job execution service meta.
   *
   * @param serviceName Name of the service.
   * @param routeContext Route context for service app.
   * @param serviceUrl Qualified url for service app.
   * @param refPackage Project base package.
   * @param serviceAnnotation Service annotation.
   * @return job execution service from service jar scanning.
   */
  public ServiceMeta generateJesMeta(String serviceName, String routeContext, String serviceUrl,
      String refPackage, Class<? extends Annotation> serviceAnnotation) {
    ServiceMeta serviceMeta = new ServiceMeta(serviceName);
    serviceMeta.setExecutionMode(ExecutionMode.JOB);
    serviceMeta.setRoute(new ServiceRoute(routeContext, serviceUrl));
    addInterfaceMeta(JobExecService.class, serviceMeta);
    addJobDetails(serviceName, refPackage, serviceUrl);
    addServiceDetails(serviceAnnotation, serviceMeta, refPackage);
    addSubscriberDetails(serviceMeta, refPackage);
    return serviceMeta;
  }



  /**
   * Generate {@link JobMeta} from service jar and store into {@link JobMetaRegistry}.
   * 
   * @param serviceName Name of the service.
   * @param refPackage base package of a service
   */
  private void addJobDetails(String serviceName, String refPackage, String serviceUrl) {
    refPackage = refPackage == null ? "" : refPackage;
    Set<Class<?>> jobClasses = getClassPathAnalyser(refPackage).getAnnotatedTypes(Job.class);
    for (Class<?> jobClass : jobClasses) {
      Job annotatedJob = jobClass.getAnnotation(Job.class);
      if (annotatedJob != null) {
        JobMeta jobMeta = new JobMeta();
        jobMeta.setAffinity(annotatedJob.affinity());
        jobMeta.setQualifiedName(jobClass.getCanonicalName());
        jobMeta.setJobExecServiceAddress(serviceUrl + serviceName + "/jes/execJob");

        getJobMetaRegistry().registerJobMeta(jobClass.getCanonicalName(), jobMeta);
      }
    }
  }

  /**
   * Add meta details such as interfaces, operations etc to service meta.
   * 
   * @param serviceAnnotation Service annotation which is to be scanned.
   * @param serviceMeta Meta instance in which details are to be added.
   * @param refPackage base package of a service
   */
  private void addServiceDetails(Class<? extends Annotation> serviceAnnotation,
      ServiceMeta serviceMeta, String refPackage) {
    refPackage = refPackage == null ? "" : refPackage;
    Set<Class<?>> serviceInterfaces =
        getClassPathAnalyser(refPackage).getAnnotatedTypes(serviceAnnotation);
    for (Class<?> serviceInterface : serviceInterfaces) {
      if (serviceInterface.isInterface()) {
        addInterfaceMeta(serviceInterface, serviceMeta);
      }
    }
  }

  /**
   * Add subscriber details as service url, service name, operations, event type.
   * 
   * @param serviceMeta Service method.
   * @param refPackage base package of a service
   */
  private void addSubscriberDetails(ServiceMeta serviceMeta, String refPackage) {
    refPackage = refPackage == null ? "" : refPackage;
    Set<Class<?>> serviceAlternateInterfaces =
        getClassPathAnalyser(refPackage).getAnnotatedTypes(SubscriberListener.class);

    for (Class<?> serviceAlternateInterface : serviceAlternateInterfaces) {
      if (serviceAlternateInterface.isInterface()) {
        registerSubscriberMeta(serviceAlternateInterface, serviceMeta);
      }
    }
  }

  /**
   * Registers subscriber data.
   * 
   * @param clazz Class definition of service interface.
   * @param serviceMeta Service method.
   */
  private void registerSubscriberMeta(Class<?> clazz, ServiceMeta serviceMeta) {
    for (Method method : getClassPathAnalyser().getAnnotatedMethods(clazz, ServiceOp.class)) {
      registerSubscriber(method, serviceMeta);
    }
  }

  /**
   * Registers subscriber meta into subscriber registry.
   * 
   * @param method Service method.
   * @param serviceMeta Service method.
   */
  private void registerSubscriber(Method method, ServiceMeta serviceMeta) {
    Parameter parameter=null;
    ServiceOp annotation = method.getAnnotation(ServiceOp.class);
    EventSubscriber baseSubscriber = new EventSubscriber();
    baseSubscriber.setServiceUrl(serviceMeta.getRoute().getServiceUrl());
    baseSubscriber.setServiceName(serviceMeta.getName());
    Subscriber subscriber = new Subscriber();
    if (method.getParameterCount() != 0) {
     parameter = method.getParameters()[0];
     subscriber.setParameterName(parameter.getName());
     subscriber.setParameterType(parameter.getType().getCanonicalName());
    }
    subscriber.setMethodName(annotation.path());
    baseSubscriber.setSubscriber(subscriber);
    getServiceBus().register(baseSubscriber);

  }

  /**
   * Add service interface meta information to service meta.
   * 
   * @param clazz Class definition of service interface.
   * @param serviceMeta Meta instance in which details are to be added.
   */
  private void addInterfaceMeta(Class<?> clazz, ServiceMeta serviceMeta) {
    InterfaceMeta interfaceMeta = new InterfaceMeta();
    interfaceMeta.setName(clazz.getSimpleName());
    interfaceMeta.setQualifiedClassName(clazz.getCanonicalName());
    interfaceMeta.setParent(serviceMeta.lightweightCopy());
    serviceMeta.addInterface(interfaceMeta);

    for (Method method : getClassPathAnalyser().getAnnotatedMethods(clazz, ServiceOp.class)) {
      addOperation(method, interfaceMeta);
    }
  }

  /**
   * Adds service operations under interface meta.
   * 
   * @param method Service method.
   * @param interfaceMeta Interface under which service ops are to be added.
   */
  private void addOperation(Method method, InterfaceMeta interfaceMeta) {
    ServiceOp annotation = method.getAnnotation(ServiceOp.class);
    ServiceOpMeta serviceOpMeta = new ServiceOpMeta();
    serviceOpMeta.setFriendly(annotation.friendly());
    serviceOpMeta.setMethod(annotation.method());
    serviceOpMeta.setPath(annotation.path());
    serviceOpMeta.setName(method.getName());
    serviceOpMeta.setAsyncFlag(annotation.async());
    serviceOpMeta.setResultTypeName(method.getReturnType().getCanonicalName());
    serviceOpMeta.setParent(interfaceMeta.lightweightCopy());
    addParameters(method.getParameters(), serviceOpMeta);
    interfaceMeta.addOperation(serviceOpMeta);

  }

  /**
   * Add parameter information to service operation.
   * 
   * @param parameters Array of parameter definitions.
   * @param serviceOpMeta Service operation in which parameters are to be added.
   */
  private void addParameters(Parameter[] parameters, ServiceOpMeta serviceOpMeta) {
    for (int i = 0; i < parameters.length; i++) {
      Parameter parameter = parameters[i];
      org.appops.core.service.Parameter param =
          new org.appops.core.service.Parameter(i, parameter.getName());
      param.setTypeName(parameter.getType().getCanonicalName());
      serviceOpMeta.addParameter(param);
    }
  }

  /**
   * <p>
   * Getter for the field <code>classPathAnalyser</code>.
   * </p>
   *
   * @return a {@link org.appops.core.ClassPathAnalyser} object.
   */
  public ClassPathAnalyser getClassPathAnalyser() {
    return getClassPathAnalyser("");
  }

  /**
   * It returns the instance of {@link org.appops.core.ClassPathAnalyser}.
   *
   * @param refPackage base package of a service
   * @return instance of {@link org.appops.core.ClassPathAnalyser}
   */
  public ClassPathAnalyser getClassPathAnalyser(String refPackage) {
    if (classPathAnalyser == null) {
      classPathAnalyser = new ClassPathAnalyser(refPackage);
      classPathAnalyser.setTypeScanner(new TypeScanner());
    }
    return classPathAnalyser;
  }

  /**
   * <p>
   * Setter for the field <code>classPathAnalyser</code>.
   * </p>
   *
   * @param classPathAnalyser a {@link org.appops.core.ClassPathAnalyser} object.
   */
  public void setClassPathAnalyser(ClassPathAnalyser classPathAnalyser) {
    this.classPathAnalyser = classPathAnalyser;
  }


  /**
   * <p>
   * Getter for the field <code>jobMetaRegistry</code>.
   * </p>
   *
   * @return a {@link org.appops.service.store.JobMetaRegistry} object.
   */
  public JobMetaRegistry getJobMetaRegistry() {
    return jobMetaRegistry;
  }

  /**
   * <p>
   * Setter for the field <code>jobMetaRegistry</code>.
   * </p>
   *
   * @param jobMetaRegistry a {@link org.appops.service.store.JobMetaRegistry} object.
   */
  @Inject
  public void setJobMetaRegistry(JobMetaRegistry jobMetaRegistry) {
    this.jobMetaRegistry = jobMetaRegistry;
  }

  /**
   * <p>
   * Getter for the field <code>serviceBus</code>.
   * </p>
   *
   * @return a {@link org.appops.cache.slim.ServiceBus} object.
   */
  public ServiceBus getServiceBus() {
    return serviceBus;
  }

  /**
   * <p>
   * Setter for the field <code>serviceBus</code>.
   * </p>
   *
   * @param subscriberRegistry a {@link org.appops.cache.slim.ServiceBus} object.
   */
  @Inject
  public void setServiceBus(ServiceBus subscriberRegistry) {
    this.serviceBus = subscriberRegistry;
  }

}


