/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */
package org.appops.service.invocation;

import java.util.Set;
import org.appops.core.ClassPathAnalyser;
import org.appops.core.TypeScanner;
import org.appops.core.annotation.AsyncOf;

/**
 * A class which is responsible to find the async implementation.
 *
 */
public class FindAyncImplementation {

  private ClassPathAnalyser classPathAnalyser;

  /**
   * It returns the async implementation of given class.
   * 
   * @param apiClass an api class
   * @return async implemented class
   */
  public Class<?> getAsyncImplementation(Class<?> apiClass) {

    String packageName = apiClass.getPackage().getName();
    Set<Class<?>> interfaceList =
        getClassPathAnalyser(packageName).getAnnotatedTypes(AsyncOf.class);
    for (Class clazz : interfaceList) {
      AsyncOf asyncOfAnnoration = (AsyncOf) clazz.getAnnotation(AsyncOf.class);
      Class syncClazz = asyncOfAnnoration.sync();
      String syncClazzName = syncClazz.getName();
      if (apiClass.getName().equals(syncClazzName)) {
        return clazz;
      }
    }
    return null;
  }

  /**
   * It returns the instance of {@link ClassPathAnalyser} of given reference package.
   * 
   * @param refPackage reference package name
   * @return instance of {@link ClassPathAnalyser}
   */
  public ClassPathAnalyser getClassPathAnalyser(String refPackage) {
    if (classPathAnalyser == null) {
      classPathAnalyser = new ClassPathAnalyser(refPackage);
      classPathAnalyser.setTypeScanner(new TypeScanner());
    }
    return classPathAnalyser;
  }

}
