/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.service.job;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.appops.core.ClassPathAnalyser;
import org.appops.core.annotation.JobEntry;
import org.appops.core.job.token.JobToken;
import org.appops.log.service.slim.service.Level;
import org.appops.logging.impl.LogManager;
import org.appops.logging.logger.Logger;
import org.appops.marshaller.DescriptorType;
import org.appops.marshaller.Marshaller;
import org.appops.service.crypto.ContentCrypto;
import org.appops.service.exception.InvocationException;
import org.appops.service.job.pool.JobPool;
import com.google.inject.Inject;
import com.google.inject.Injector;

/**
 * Processor class which processes a single job token to execute job. This class is to be used in
 * JobExecutorService entry point.
 *
 * @author deba
 * @version $Id: $Id
 */
public class JobExecServiceImpl implements JobExecService {

  @Inject
  private Injector injector;
  private JobPool pool;
  private Logger logger;

  /**
   * <p>
   * Constructor for JobExecServiceImpl.
   * </p>
   */
  protected JobExecServiceImpl() {

  }

  @Inject
  /**
   * <p>
   * Constructor for JobExecServiceImpl.
   * </p>
   *
   * @param logManager a {@link org.appops.logging.impl.LogManager} object.
   */
  protected JobExecServiceImpl(LogManager logManager) {
    logger = logManager.getRootLogger();
  }

  /**
   * {@inheritDoc}
   *
   * Decrypt job token and executes job by using job token provided.
   */
  @Override
  public void executeJob(String token) {
    try {
      JobToken jobToken = decryptToken(token);

      if (jobToken != null) {
        Class<?> jobClazz = Class.forName(jobToken.getJobClassName());
        Object jobClassObject = injector.getInstance(jobClazz);
        Method method = getMethod(jobClazz);
        List<Object> params = getParameters(method, jobToken.getParameters());
        logger.withLevel(Level.FINE)
            .withMessage("Token submitted for execution :" + jobToken.getGuid());
        getPool().submitJob(jobClassObject, params.toArray(), jobToken);
      }
    } catch (Exception e) {
      logger.withLevel(Level.SEVERE)
          .withMessage("Exception occured in " + getClass().getCanonicalName() + "::executeJob() ")
          .withMeta("Exception", e);
      throw new JobExecutionException(e);
    }
  }


  /**
   * Find method from provided job class whcih have {@link JobEntry} annotation.if more than one
   * method has {@link JobEntry} annotation then it will throw an exception.
   * 
   * @param jobClazz job class name which operation to be invoke.
   * @return method which have {@link JobEntry} annotation.
   */
  private Method getMethod(Class<?> jobClazz) {
    List<Method> methods =
        injector.getInstance(ClassPathAnalyser.class).getAnnotatedMethods(jobClazz, JobEntry.class);
    if (methods.isEmpty()) {
      throw new InvocationException(
          "Method must have " + JobEntry.class.getName() + " annotation for execution.");
    }
    if (methods.size() > 1) {
      throw new InvocationException(
          "Found more than one method with " + JobEntry.class.getName() + " annotation.");
    }

    return methods.get(0);
  }

  /**
   * Populate and return parameters which need to actual method invocation.
   * 
   * @param method method which to be invoke.
   * @param userParameters job token parameter map.
   * @return list of an object with actual value and type.
   */
  private List<Object> getParameters(Method method, Map<String, Serializable> userParameters) {
    List<Object> paramsToReturn = new ArrayList<>();

    Parameter[] methodParameters = method.getParameters();
    for (Parameter methodParam : methodParameters) {
      String name = methodParam.getName();
      Object value = userParameters.get(name);
      if (value != null) {
        Class<?> typeClazz = methodParam.getType();
        value = getType(typeClazz, value);
        paramsToReturn.add(value);
      } else {
        paramsToReturn.add(null);
      }
    }
    return paramsToReturn;
  }

  /**
   * Locate type of a provided value.
   * 
   * @param typeClazz type of the value.
   * @param value actual value which to be store in into appropriate type.
   * @return Result with actual type.
   * 
   */
  private Object getType(Class<?> typeClazz, Object value) {
    if (!typeClazz.isAssignableFrom(value.getClass())) {
      String stringVal;
      Marshaller marshaller = injector.getInstance(Marshaller.class);
      if (value instanceof String) {
        stringVal = (String) value;
        if (typeClazz.isEnum()) {
          value = Enum.valueOf((Class<Enum>) typeClazz, stringVal);
          stringVal = null;
        }
      } else {
        stringVal = marshaller.marshall(value, DescriptorType.JSON);
      }
      if (stringVal != null) {
        value = marshaller.unmarshall(stringVal, typeClazz, DescriptorType.JSON);
      }
    }
    return value;
  }

  /**
   * Decrypt token and populate instance of {@link JobToken} for excution.
   * 
   * @param token encrypted token which to be derypt.
   * @return instance of {@link JobToken}
   */
  private JobToken decryptToken(String token) {
    String tokenJson = new ContentCrypto().decrypt(token);
    JobToken jobToken = injector.getInstance(Marshaller.class).unmarshall(tokenJson, JobToken.class,
        DescriptorType.JSON);
    return jobToken;
  }

  /**
   * <p>
   * Getter for the field <code>pool</code>.
   * </p>
   *
   * @return a {@link org.appops.service.job.pool.JobPool} object.
   */
  public JobPool getPool() {
    return pool;
  }

  /**
   * <p>
   * Setter for the field <code>pool</code>.
   * </p>
   *
   * @param pool a {@link org.appops.service.job.pool.JobPool} object.
   */
  @Inject
  public void setPool(JobPool pool) {
    this.pool = pool;
  }

}
