/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.service.impl.store;

import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import org.appops.core.service.OpParameterMap;
import org.appops.core.service.Parameter;
import org.appops.core.service.ServiceRoute;
import org.appops.core.service.meta.InterfaceMeta;
import org.appops.core.service.meta.ServiceMeta;
import org.appops.core.service.meta.ServiceOpMeta;
import org.appops.log.service.slim.service.Level;
import org.appops.logging.logger.Logger;
import org.appops.service.impl.ServiceStoreException;
import org.appops.service.store.ServiceMetaRegistry;
import org.appops.slim.base.api.ServiceMetaManager;
import org.appops.slim.base.core.ServiceStore;

public class ServiceMetaManagerImpl implements ServiceMetaManager {

  private ServiceMetaRegistry serviceMetaRegistry;

  private Logger logger;

  @Inject
  public ServiceMetaManagerImpl(ServiceMetaRegistry serviceMetaRegistry) {
    this.serviceMetaRegistry = serviceMetaRegistry;
  }

  @Override
  public void registerServiceMeta(ServiceMeta serviceMeta) {
    serviceMetaRegistry.addServiceMeta(serviceMeta);
  }

  @Override
  public ServiceMeta getServiceMeta(String name) {
    ServiceMeta meta = getServiceMetaRegistry().getServiceMeta(name);
    return meta;
  }

  @Override
  public ServiceRoute getServiceRoute(String name) {
    ServiceMeta meta = getServiceMetaRegistry().getServiceMeta(name);
    return meta.getRoute();
  }

  @Override
  public ServiceOpMeta getOpMeta(String path) {
    return getServiceMetaRegistry().getOpMeta(path);
  }

  @Override
  public ServiceOpMeta getOpMeta(String service, String path) {
    return getServiceMetaRegistry().getOpMeta(service, path);
  }

  @Override
  public ServiceOpMeta getOpMeta(Long id) {
    return getServiceMetaRegistry().getOpMetaById(id);
  }

  public ServiceMetaRegistry getServiceMetaRegistry() {
    return serviceMetaRegistry;
  }

  @Override
  public Collection<ServiceMeta> readAllServices() {
    if (isLogServiceStarted()) {
      saveOpLog(ServiceStore.class.getSimpleName(), "readAllServices");
    }
    return getServiceMetaRegistry().getAllServiceMetas();
  }

  private boolean isLogServiceStarted() {
    boolean isLogServiceExists = false;

    Collection<ServiceMeta> serviceMetas = getServiceMetaRegistry().getAllServiceMetas();
    for (ServiceMeta serviceMeta : serviceMetas) {
      isLogServiceExists = serviceMeta.getName().equals("LogService");
      if (isLogServiceExists) {
        break;
      }
    }
    return isLogServiceExists;
  }

  @Override
  public Collection<ServiceMeta> getServicesByPage(Integer startIndex, Integer pageSize) {
    if (isLogServiceStarted()) {
      saveOpLog(ServiceStore.class.getSimpleName(), "getServicesByPage");
    }
    return getServiceMetaRegistry().getServicesByPage(startIndex, pageSize);
  }

  @Override
  public Collection<InterfaceMeta> getServiceInterfaces(String serviceName) {
    if (isLogServiceStarted()) {
      saveOpLog(ServiceStore.class.getSimpleName(), "getServiceInterfaces");
    }
    return getServiceMetaRegistry().getInterfaces(serviceName);
  }

  @Override
  public Collection<ServiceOpMeta> getOperations(String serviceName, String interfaceName) {
    return getServiceMetaRegistry().getOperations(serviceName, interfaceName);
  }

  @Override
  public Collection<ServiceMeta> getFilteredServices(String filter) {
    try {
      return filterServices(filter, readAllServices(), true);
    } catch (Exception e) {
      throw new ServiceStoreException("Failed to filter services using keyword(s) ->" + filter);
    }
  }

  @Override
  public ArrayList<Parameter> getOperationParameters(String serviceName, String interfaceName,
      String operationName) {
    return getServiceMetaRegistry().getParameters(serviceName, interfaceName, operationName);
  }



  private Collection<ServiceMeta> filterServices(String filter, Collection<ServiceMeta> services,
      boolean deepFilter) {
    Set<ServiceMeta> listToReturn = new HashSet<>();
    for (ServiceMeta service : services) {
      if (!service.getName().contains(filter)) {
        continue;
      }
      ServiceMeta filteredCopy = service.lightweightCopy();
      if (deepFilter) {
        filteredCopy.addInterfaces(filterInterfaces(filter, service.getInterfaces(), deepFilter));
      }
      listToReturn.add(filteredCopy);
    }
    return listToReturn;

  }

  private Collection<InterfaceMeta> filterInterfaces(String filter,
      Collection<InterfaceMeta> interfaces, boolean deepFilter) {
    Set<InterfaceMeta> listToReturn = new HashSet<>();
    for (InterfaceMeta interf : interfaces) {
      if (!interf.getName().contains(filter)) {
        continue;
      }
      InterfaceMeta filteredCopy = interf.lightweightCopy();
      if (deepFilter) {
        filteredCopy.addOperations(filterOperations(filter, interf.getOperations(), deepFilter));
      }
      listToReturn.add(filteredCopy);
    }

    return listToReturn;
  }

  private Collection<ServiceOpMeta> filterOperations(String filter, Set<ServiceOpMeta> operations,
      boolean deepFilter) {
    Set<ServiceOpMeta> operationList = new HashSet<>();
    for (ServiceOpMeta operation : operations) {
      if (!operation.getName().contains(filter)) {
        continue;
      }
      ServiceOpMeta filteredCopy = operation.lightweightCopy();
      if (deepFilter) {
        filteredCopy.setParameters(filterParameters(filter, operation.getParameterSet()));
      }
      operationList.add(filteredCopy);
    }
    return operationList;
  }

  private OpParameterMap filterParameters(String filter, Set<Parameter> parameterSet) {
    OpParameterMap parameterMap = new OpParameterMap();
    for (Parameter parameter : parameterSet) {
      if (!parameter.getName().contains(filter)) {
        continue;
      }
      parameterMap.addParameter(parameter);
    }
    return parameterMap;
  }

  private void saveOpLog(String serviceName, String opName) {
    String logMessage = "Service operation log :  Service_Name -> " + serviceName
        + ", OperationName -> " + opName + " , Detailed message :- Started operation execution";
    logger.withLevel(Level.INFO).withMessage(logMessage).log();
  }

  public Logger getRootLogger() {
    return logger;
  }

  @Inject
  public void setRootLogger(Logger rootLogger) {
    this.logger = rootLogger;
  }

}
