/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.service.impl.store;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Table;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;
import org.appops.core.service.ExecutionMode;
import org.appops.core.service.OpParameterMap;
import org.appops.core.service.OpPathMatcher;
import org.appops.core.service.Parameter;
import org.appops.core.service.meta.InterfaceMeta;
import org.appops.core.service.meta.ServiceMeta;
import org.appops.core.service.meta.ServiceOpMeta;
import org.appops.core.service.signature.ServiceOpSignatureBuilder;
import org.appops.service.impl.ServiceStoreException;
import org.appops.service.store.ServiceMetaRegistry;


public class ServiceMetaRegistryImpl implements ServiceMetaRegistry {

  private Table<String, ExecutionMode, ServiceMeta> serviceMetaTable = HashBasedTable.create();
  private ServiceOpSignatureBuilder signatureBuilder;
  private static Long serviceId = 0L;

  @Override
  public ServiceMeta getServiceMeta(String name) {
    if (!isServiceExists(name)) {
      throw new ServiceStoreException("Service with name -> " + name + " does not exist.");
    }
    ServiceMeta serviceMeta = serviceMetaTable.get(name, ExecutionMode.SERVICE);
    if (serviceMeta != null) {
      return serviceMeta;
    } else {
      return serviceMetaTable.get(name, ExecutionMode.JOB);
    }
  }

  /**
   * Checks if service with name exist in store or not.
   * 
   * @param name Name of service to be looked for.
   * @return true if service already exists in store, false otherwise.
   */
  public boolean isServiceExists(String name) {
    if (name == null || name.isEmpty()) {
      throw new ServiceStoreException("Null or empty service name passed with service meta");
    }
    return serviceMetaTable.containsRow(name);
  }

  @Override
  public InterfaceMeta getInterfaceMeta(String service, String name) {
    if (name == null || name.isEmpty()) {
      throw new ServiceStoreException(
          "Null or empty interface name passed, cannot add interfaceMeta.");
    }

    for (InterfaceMeta intrfcMeta : getServiceMeta(service).getInterfaces()) {
      if (intrfcMeta.getName().equals(name)) {
        return intrfcMeta;
      }
    }
    throw new ServiceStoreException(
        "Interface with name -> " + name + " does not exist under service ->" + service);
  }

  @Override
  public ServiceOpMeta getOpMeta(String pathOrSignature) {
    if (pathOrSignature == null) {
      throw new ServiceStoreException("Null Path / Signature passed, cannot find service op meta");
    }

    for (ServiceMeta serviceMeta : serviceMetaTable.values()) {
      for (InterfaceMeta intrfcMeta : serviceMeta.getInterfaces()) {
        for (ServiceOpMeta opMeta : intrfcMeta.getOperations()) {
          if (matchPath(opMeta.getPath(), pathOrSignature) || matchSign(opMeta, pathOrSignature)) {
            return opMeta;
          }
        }
      }
    }
    throw new ServiceStoreException(
        "Operation with path/signature -> " + pathOrSignature + " does not exist.");
  }

  @Override
  public ServiceOpMeta getOpMeta(String service, String pathOrSignature) {
    if (pathOrSignature == null) {
      throw new ServiceStoreException("Null Path / Signature passed, cannot find service op meta");
    }

    for (ServiceMeta serviceMeta : serviceMetaTable.values()) {
      if (!serviceMeta.getName().equals(service)) {
        continue;
      }
      for (InterfaceMeta intrfcMeta : serviceMeta.getInterfaces()) {
        for (ServiceOpMeta opMeta : intrfcMeta.getOperations()) {
          if (matchPath(opMeta.getPath(), pathOrSignature) || matchSign(opMeta, pathOrSignature)) {
            return opMeta;
          }
        }
      }
    }
    throw new ServiceStoreException(
        "Operation with path/signature -> " + pathOrSignature + " does not exist.");
  }

  private boolean matchSign(ServiceOpMeta opMeta, String expectedSign) {
    String opSign = getSignatureBuilder().buildOpSignature(opMeta);
    return (expectedSign != null && opSign != null && opSign.equals(expectedSign));
  }

  private boolean matchPath(String opPath, String requestPath) {
    return new OpPathMatcher().matchPath(opPath, requestPath);
  }

  @Override
  public ServiceOpMeta getOpMetaById(Long id) {
    if (id == null) {
      throw new ServiceStoreException("Null id passed, cannot find service op meta");
    }
    for (ServiceMeta serviceMeta : serviceMetaTable.values()) {
      for (InterfaceMeta intrfcMeta : serviceMeta.getInterfaces()) {
        for (ServiceOpMeta opMeta : intrfcMeta.getOperations()) {
          if (opMeta.getId() == id) {
            return opMeta;
          }
        }
      }
    }
    throw new ServiceStoreException("Operation with id -> " + id + " does not exist.");
  }

  @Override
  public void addServiceMeta(ServiceMeta serviceMeta) {
    if (serviceMeta == null) {
      throw new ServiceStoreException("Cannot store null service meta");
    }
    if (isServiceExists(serviceMeta.getName())) {
      // throw new ServiceStoreException("Service with name -> " + serviceMeta.getName()
      // + " already exists. Duplicate service is not allowed.");
      return;
    } else {
      serviceMeta.setId(++serviceId);
      ExecutionMode executionMode = serviceMeta.getExecutionMode() == null ? ExecutionMode.SERVICE
          : serviceMeta.getExecutionMode();
      serviceMetaTable.put(serviceMeta.getName(), executionMode, serviceMeta);
    }

  }

  @Override
  public void addInterfaceMeta(String service, InterfaceMeta intrfcMeta) {
    if (intrfcMeta == null) {
      throw new ServiceStoreException(
          "Cannot store null interface meta under service -> " + service);
    }
    getServiceMeta(service).addInterface(intrfcMeta);

  }

  @Override
  public void addOpMeta(String service, String intrfc, ServiceOpMeta opMeta) {
    if (opMeta == null) {
      throw new ServiceStoreException(
          "Cannot store null service op meta under service -> " + service);
    }
    getInterfaceMeta(service, intrfc).addOperation(opMeta);
  }

  public ServiceOpSignatureBuilder getSignatureBuilder() {
    return signatureBuilder;
  }

  @Inject
  public void setSignatureBuilder(ServiceOpSignatureBuilder signatureBuilder) {
    this.signatureBuilder = signatureBuilder;
  }

  @Override
  public Collection<ServiceMeta> getAllServiceMetas() {
    return serviceMetaTable.values();
  }

  /**
   * Fetches service entries by page.
   * 
   * @param startIndex Start index of page.
   * @param pageSize Number of records to be included in page.
   * 
   * @return returns collection of services included in page.
   */
  public Collection<ServiceMeta> getServicesByPage(Integer startIndex, Integer pageSize) {

    try {
      List<ServiceMeta> listToReturn = new ArrayList<ServiceMeta>();

      int endIndex = startIndex + pageSize;

      Collection<ServiceMeta> mockData = getAllServiceMetas();
      Iterator<ServiceMeta> iterator = mockData.iterator();

      while (iterator.hasNext()) {

        ServiceMeta element = iterator.next();

        if (element.getId() >= startIndex && element.getId() <= endIndex) {

          if (mockData.size() != startIndex) {
            listToReturn.add(element);
          }
        }
      }

      return listToReturn;
    } catch (NullPointerException e) {
      throw new ServiceStoreException(e.getMessage());
    }

  }

  @Override
  public Collection<InterfaceMeta> getInterfaces(String serviceName) {
    return getServiceMeta(serviceName).getInterfaces();
  }

  @Override
  public Collection<ServiceOpMeta> getOperations(String serviceName, String interfaceName) {
    Set<ServiceOpMeta> opMetas = new HashSet<>();
    for (InterfaceMeta interfaceMeta : getInterfaces(serviceName)) {
      if (interfaceMeta.getName().equals(interfaceName)) {
        opMetas.addAll(interfaceMeta.getOperations());
        break;
      }
    }
    return opMetas;
  }

  @Override
  public ArrayList<Parameter> getParameters(String serviceName, String interfaceName,
      String operationName) {
    OpParameterMap opParams = new OpParameterMap();
    ArrayList<Parameter> paramList = new ArrayList<Parameter>();
    Collection<InterfaceMeta> interfaceMetaSet = getInterfaces(serviceName);
    for (InterfaceMeta interfaceMeta : interfaceMetaSet) {
      if (interfaceMeta.getName().equals(interfaceName)) {
        Set<ServiceOpMeta> serviceOpmetaSet = interfaceMeta.getOperations();
        for (ServiceOpMeta opMeta : serviceOpmetaSet) {
          if (opMeta.getName().equals(operationName)) {
            OpParameterMap opMap = opMeta.getParameters();
            for (Entry<Integer, Parameter> entry : opMap.entrySet()) {
              paramList.add(entry.getValue());
            }
            break;
          }
        }
      }
    }
    return paramList;
  }

}
