/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.service.impl.store;

import org.appops.core.job.JobMeta;
import org.appops.core.job.token.JobToken;
import org.appops.core.job.token.JobTokenSet;
import org.appops.core.mime.MimeType;
import org.appops.core.service.OpParameterMap;
import org.appops.log.service.slim.service.Level;
import org.appops.logging.impl.LogManager;
import org.appops.logging.logger.Logger;
import org.appops.marshaller.DescriptorType;
import org.appops.marshaller.Marshaller;
import org.appops.service.crypto.ContentCrypto;
import org.appops.service.exception.InvocationException;
import org.appops.service.impl.ServiceStoreException;
import org.appops.service.job.TokenRouter;
import org.appops.service.store.JobMetaRegistry;
import org.appops.web.common.client.ServerResponse;
import org.appops.web.common.client.WebClient;
import com.google.inject.Inject;


/**
 * Locates job execution service from job meta information stored in service store and routes the
 * job token to job execution service.
 */
public class TokenRouterImpl implements TokenRouter {
  private JobMetaRegistry jobMetaRegistry;
  private WebClient<?> webClient;
  private Marshaller marshaller;
  private Logger logger;

  public TokenRouterImpl() {

  }

  @Inject
  public TokenRouterImpl(LogManager logManager) {
    logger = logManager.getRootLogger();
  }

  /**
   * Locates and routes job token to appropriate job execution service.
   * 
   * @param token Job token to be routed.
   */
  public void routeJobToken(JobToken token) {
    if (token != null) {
      JobMeta jobMeta = getJobMetaRegistry().getJobMeta(token.getJobKey());
      String restUrl = null;
      if (jobMeta != null) {
        token.setJobAffinity(jobMeta.getAffinity());
        restUrl = jobMeta.getJobExecServiceAddress();
      }
      String opParamJson = populateOpParamMapJson(token);

      ServerResponse serverResponse = getWebClient().post(restUrl, opParamJson, MimeType.JSON);
      if (serverResponse.getStatus() != 200) {
        logger.withLevel(Level.SEVERE).withMessage("Rest invocation failed :" + restUrl)
            .withMeta("ErrorCode", serverResponse.getStatus()).log();
        throw new InvocationException("Rest invocation failed, url -> " + restUrl
            + ", error code -> " + serverResponse.getStatus());
      }
    }
  }

  /**
   * Locates and routes job tokens to appropriate job execution service.
   * 
   * @param tokens Set of {@link JobToken} to be routed.
   */
  @Override
  public void routeJobTokens(JobTokenSet tokens) {
    for (JobToken token : tokens) {
      try {
        routeJobToken(token);
      } catch (Exception e) {
        throw new ServiceStoreException(e);
      }
    }
  }

  /**
   * Populate {@link OpParameterMap } from job token and encrypt {@link OpParameterMap }.
   * 
   * @param token job token which to be process.
   * @return encrypted job token .
   */
  private String populateOpParamMapJson(JobToken token) {

    String tokenJson = getMarshaller().marshall(token, DescriptorType.JSON);
    tokenJson = new ContentCrypto().encrypt(tokenJson);

    OpParameterMap map = new OpParameterMap();
    map.addParameter(0, "token", tokenJson);

    return getMarshaller().marshall(map, DescriptorType.JSON);
  }

  public JobMetaRegistry getJobMetaRegistry() {
    return jobMetaRegistry;
  }

  @Inject
  public void setJobMetaRegistry(JobMetaRegistry jobMetaRegistry) {
    this.jobMetaRegistry = jobMetaRegistry;
  }

  public WebClient<?> getWebClient() {
    return webClient;
  }

  @Inject
  public void setWebClient(WebClient webClient) {
    this.webClient = webClient;
  }

  public Marshaller getMarshaller() {
    return marshaller;
  }

  @Inject
  public void setMarshaller(Marshaller marshaller) {
    this.marshaller = marshaller;
  }

}
