/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.slim.base.invocation;

import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.name.Named;
import java.util.Collection;
import org.appops.core.constant.CoreGuiceConstant;
import org.appops.core.deployment.DeploymentConfig;
import org.appops.core.mime.MimeType;
import org.appops.core.service.OpParameterMap;
import org.appops.core.service.Parameter;
import org.appops.core.service.RequestMethod;
import org.appops.core.service.ServiceRoute;
import org.appops.core.service.meta.ServiceOpMeta;
import org.appops.marshaller.DescriptorType;
import org.appops.marshaller.Marshaller;
import org.appops.slim.base.api.ServiceMetaManager;
import org.appops.slim.base.exception.InvocationException;
import org.appops.web.common.client.ServerResponse;
import org.appops.web.common.client.WebClient;

/**
 * Sends a rest request using http connection to service deployed in stand alone mode and converts
 * response data into appropriate type.
 *
 * @author deba
 * @version $Id: $Id
 */
public class ApiRestInvoker {

  private Provider<DeploymentConfig> currentDeployment;
  private Provider<WebClient> webClient;
  private Marshaller marshaller;
  private Provider<ServiceMetaManager> serviceStore;

  /**
   * Sends rest request to service API and fetches result if expected.
   *
   * @param opMeta Meta information of an operation which is to be invoked.
   * @return Result obtained from rest call.
   */
  public Object invokeRest(ServiceOpMeta opMeta) {
    try {
      String restUrl = getBaseRestUrl(opMeta);
      WebClient<?> webClient = getWebClient();
      ServerResponse serverResponse = null;
      if (RequestMethod.GET.equals(opMeta.getMethod())) {
        restUrl = buildUrl(restUrl, opMeta, true);
        serverResponse = webClient.get(restUrl);
      } else if (RequestMethod.POST.equals(opMeta.getMethod())) {
        restUrl = buildUrl(restUrl, opMeta, false);
        String parameterJson = createParameterJson(opMeta.getParameters());
        serverResponse = webClient.post(restUrl, parameterJson, MimeType.JSON);
      } else {
        throw new InvocationException(
            "No request method provided in operation meta, please provide - GET/POST");
      }
      if (serverResponse.getStatus() != 200) {
        throw new InvocationException("Rest invocation failed, url -> " + restUrl
            + ", error code -> " + serverResponse.getStatus());
      }
      String responseText = serverResponse.getContentAsString();
      if (responseText == null || responseText.isEmpty()) {
        return "";
      }
      return convertResult(responseText, opMeta.getResultTypeName());
    } catch (Exception e) {
      throw ((e instanceof InvocationException) ? (InvocationException) e
          : new InvocationException(e));
    }

  }


  private String getBaseRestUrl(ServiceOpMeta opMeta) {
    String deploymentUrl = getCurrentDeployment().getUrl();
    if (opMeta.serviceName().equals("ServiceStore")) {
      return getCurrentDeployment().getUrl();
    }
    ServiceRoute serviceRoute = getServiceStore().getServiceRoute(opMeta.serviceName());
    String serviceUrl = serviceRoute != null ? serviceRoute.getServiceUrl() : "";
    return (serviceUrl != null && !serviceUrl.isEmpty()) ? serviceUrl : deploymentUrl;
  }


  private String createParameterJson(OpParameterMap parameters) {
    return marshaller.marshall(parameters, DescriptorType.JSON);
  }


  private Object convertResult(String contentAsString, String resultTypename) {
    if (String.class.getCanonicalName().equals(resultTypename)) {
      return contentAsString;
    }
    Class<?> resultType = null;
    try {
      resultType = Class.forName(resultTypename);
    } catch (Exception e) {
      throw new InvocationException("Error while converting rest call result", e);
    }

    return getMarshaller().unmarshall(contentAsString, resultType, DescriptorType.JSON);
  }


  /**
   * Builds complete rest url to be used to send rest request. The url also contains parameters to
   * be passed while sending request.
   *
   * @param restUrl Service url.
   * @param opMeta Meta information of service api to be called.
   * @param appendParamters append paramters
   * @return Full rest url containing all information needed.
   */
  public String buildUrl(String restUrl, ServiceOpMeta opMeta, boolean appendParamters) {
    if (!restUrl.endsWith("/")) {
      restUrl += "/";
    }

    if (opMeta.serviceName() == null || opMeta.serviceName().isEmpty()) {
      throw new InvocationException("Parent service name is null, please provide it "
          + "through service op meta instance passed.");
    }
    restUrl += /* "Op/" + */ opMeta.serviceName();

    Collection<Parameter> params = opMeta.getParameters().values();
    String friendly = opMeta.getFriendly();
    String name = opMeta.getName();
    String path = opMeta.getPath();
    if (path != null && !path.isEmpty()) {
      return restUrl += ("/" + evaluatePathParameters(path, opMeta));
    } else {
      if (friendly != null && !friendly.isEmpty()) {
        restUrl += ("/" + friendly);
      } else if (name != null && !name.isEmpty()) {
        restUrl += ("/" + name);
      } else {
        throw new InvocationException(
            "Both signature and friendly cannot be null for an operation =>" + opMeta.getName()
                + ", Service => " + opMeta.serviceName());
      }
      if (appendParamters) {
        restUrl += addRequestParameters(params);
      }
      System.out.println("ApiRestInvoker url - > " + restUrl);
      return restUrl;
    }
  }

  private String evaluatePathParameters(String path, ServiceOpMeta opMeta) {
    UrlEncodeUtil parameterEncoder = new UrlEncodeUtil();
    if (path != null && path.contains("/")) {
      String[] pathElements = path.split("/");
      for (int i = 0; i < pathElements.length; i++) {
        String pathElement = pathElements[i];
        if (pathElement.startsWith("{") && pathElement.endsWith("}")) {
          String parameterPart =
              pathElement.substring(pathElement.indexOf("{"), pathElement.indexOf("}") + 1).trim();
          String parameterName =
              parameterPart.substring(parameterPart.indexOf("{") + 1, parameterPart.indexOf("}"));
          Parameter parameter = opMeta.getParameter(parameterName);
          if (parameter != null) {
            path = path.replace(parameterPart,
                parameterEncoder.encodePathParam(parameter.getValue().toString()));
          }
        }
      }
    }
    return path;
  }


  /**
   * Builds a string containing parameters to be appended to rest url.
   * 
   * @param params Parameter information e.g. name, value etc.
   * @return String containing parameter information which is to be appended to actual request url.
   */
  private String addRequestParameters(Collection<Parameter> params) {
    String parameterPart = "";

    if (!params.isEmpty()) {
      parameterPart += "?";
    }

    for (Parameter parameter : params) {
      if (!parameterPart.endsWith("?") && !parameterPart.endsWith("&")) {
        parameterPart += "&";
      }
      parameterPart += (parameter.getName() + "=" + parameter.getValue());
    }
    return parameterPart;
  }

  /**
   * <p>
   * Getter for the field <code>currentDeployment</code>.
   * </p>
   *
   * @return a {@link org.appops.core.deployment.DeploymentConfig} object.
   */
  public DeploymentConfig getCurrentDeployment() {
    return currentDeployment.get();
  }

  /**
   * <p>
   * Setter for the field <code>currentDeployment</code>.
   * </p>
   *
   * @param currentDeployment a {@link com.google.inject.Provider} object.
   */
  @Inject
  public void setCurrentDeployment(
      @Named(CoreGuiceConstant.CURRENT_DEPLOYMENT) Provider<DeploymentConfig> currentDeployment) {
    this.currentDeployment = currentDeployment;
  }

  /**
   * <p>
   * Getter for the field <code>webClient</code>.
   * </p>
   *
   * @return a {@link org.appops.web.common.client.WebClient} object.
   */
  public WebClient getWebClient() {
    return webClient.get();
  }

  /**
   * <p>
   * Setter for the field <code>webClient</code>.
   * </p>
   *
   * @param webClient a {@link com.google.inject.Provider} object.
   */
  @Inject
  public void setWebClient(Provider<WebClient> webClient) {
    this.webClient = webClient;
  }


  /**
   * <p>
   * Getter for the field <code>marshaller</code>.
   * </p>
   *
   * @return a {@link org.appops.marshaller.Marshaller} object.
   */
  public Marshaller getMarshaller() {
    return marshaller;
  }


  /**
   * <p>
   * Setter for the field <code>marshaller</code>.
   * </p>
   *
   * @param marshaller a {@link org.appops.marshaller.Marshaller} object.
   */
  @Inject
  public void setMarshaller(Marshaller marshaller) {
    this.marshaller = marshaller;
  }


  /**
   * <p>
   * Getter for the field <code>serviceStore</code>.
   * </p>
   *
   * @return a {@link org.appops.slim.base.api.ServiceMetaManager} object.
   */
  public ServiceMetaManager getServiceStore() {
    return serviceStore.get();
  }

  /**
   * <p>
   * Setter for the field <code>serviceStore</code>.
   * </p>
   *
   * @param serviceStore a {@link com.google.inject.Provider} object.
   */
  @Inject
  public void setServiceStore(Provider<ServiceMetaManager> serviceStore) {
    this.serviceStore = serviceStore;
  }
}
