/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.slim.base.invocation;

import com.google.inject.Inject;
import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import org.appops.core.service.Parameter;
import org.appops.core.service.annotation.Service;
import org.appops.core.service.annotation.ServiceOp;
import org.appops.core.service.meta.InterfaceMeta;
import org.appops.core.service.meta.ServiceMeta;
import org.appops.core.service.meta.ServiceOpMeta;
import org.appops.core.service.signature.ServiceOpSignatureBuilder;
import org.appops.slim.base.exception.InvocationException;

/**
 * Proxy invocation handler class which takes care of service api call made using proxy.
 *
 * @author deba
 * @version $Id: $Id
 */
public class ApiProxyInvocationHandler implements InvocationHandler {

  private ApiRestInvoker apiRestInvoker;
  private ServiceOpSignatureBuilder signatureBuilder;


  /** {@inheritDoc} */
  @Override
  public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
    ServiceOpMeta opMeta = createOpMeta(method, args);
    return getApiRestInvoker().invokeRest(opMeta);
  }


  /**
   * Creates service operation meta instance from method and arguments provided.
   *
   * @param method method definition from which meta information is to be collected.
   * @param args Argument values passed to method while invocation.
   * @return operation meta instance generated.
   */
  public ServiceOpMeta createOpMeta(Method method, Object[] args) {
    ServiceOpMeta opMeta = new ServiceOpMeta();
    opMeta.setName(method.getName());
    ServiceOp opAnnotation = method.getAnnotation(ServiceOp.class);
    if (opAnnotation == null) {
      throw new InvocationException(
          "Cannot invoke operation, ServiceOp annotation is absent on method ->"
              + method.getName());
    }
    opMeta.setFriendly(opAnnotation.friendly());
    opMeta.setPath(opAnnotation.path());
    opMeta.setResultTypeName(method.getReturnType().getCanonicalName());
    opMeta.setMethod(opAnnotation.method());
    java.lang.reflect.Parameter[] parameters = method.getParameters();
    if (args != null) {
      for (int i = 0; i < args.length; i++) {
        Parameter parameter = new Parameter(i, parameters[i].getName(), args[i]);
        parameter.setTypeName(method.getParameterTypes()[i].getCanonicalName());
        opMeta.addParameter(parameter);
      }
    }
    InterfaceMeta interfaceMeta = new InterfaceMeta();
    interfaceMeta.setName(method.getDeclaringClass().getSimpleName());
    interfaceMeta.setQualifiedClassName(method.getDeclaringClass().getCanonicalName());
    opMeta.setParent(interfaceMeta);
    interfaceMeta.addOperation(opMeta);
    String serviceName = null;
    for (Annotation annotation : method.getDeclaringClass().getAnnotations()) {
      if (annotation.annotationType().isAnnotationPresent(Service.class)) {
        serviceName = annotation.annotationType().getSimpleName();
        break;
      }
    }
    if (serviceName == null) {
      throw new InvocationException(
          "Cannot invoke operation, unable to locate service name for method ->"
              + method.getName());
    }
    ServiceMeta serviceMeta = new ServiceMeta(serviceName);
    interfaceMeta.setParent(serviceMeta);
    return opMeta;
  }


  /**
   * <p>Getter for the field <code>apiRestInvoker</code>.</p>
   *
   * @return a {@link org.appops.slim.base.invocation.ApiRestInvoker} object.
   */
  public ApiRestInvoker getApiRestInvoker() {
    return apiRestInvoker;
  }


  /**
   * <p>Setter for the field <code>apiRestInvoker</code>.</p>
   *
   * @param apiRestInvoker a {@link org.appops.slim.base.invocation.ApiRestInvoker} object.
   */
  @Inject
  public void setApiRestInvoker(ApiRestInvoker apiRestInvoker) {
    this.apiRestInvoker = apiRestInvoker;
  }


  /**
   * <p>Getter for the field <code>signatureBuilder</code>.</p>
   *
   * @return a {@link org.appops.core.service.signature.ServiceOpSignatureBuilder} object.
   */
  public ServiceOpSignatureBuilder getSignatureBuilder() {
    return signatureBuilder;
  }


  /**
   * <p>Setter for the field <code>signatureBuilder</code>.</p>
   *
   * @param signatureBuilder a {@link org.appops.core.service.signature.ServiceOpSignatureBuilder} object.
   */
  @Inject
  public void setSignatureBuilder(ServiceOpSignatureBuilder signatureBuilder) {
    this.signatureBuilder = signatureBuilder;
  }

}
