/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.slim.base.invocation;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;
import org.appops.slim.base.exception.InvocationException;

/**
 * Utility class which facilitates encoding and decoding of path parameters as well as whole url.
 *
 * @author deba
 * @version $Id: $Id
 */
public class UrlEncodeUtil {

  private static final String forwardSlash = "$SLASH$";

  private Map<String, String> charReplacementMap = new HashMap<>();

  /**
   * <p>Constructor for UrlEncodeUtil.</p>
   */
  public UrlEncodeUtil() {
    charReplacementMap.put("/", forwardSlash);
  }

  /**
   * Encodes path parameter.
   *
   * @param parameter Path parameter.
   * @return Encoded parameter value.
   */
  public String encodePathParam(String parameter) {
    for (String key : charReplacementMap.keySet()) {
      parameter = parameter.contains(key)
          ? parameter.replaceAll(key, urlEncode(charReplacementMap.get(key)))
          : parameter;
    }
    return parameter;
  }

  /**
   * Decodes path parameter.
   *
   * @param parameter Path parameter.
   * @return Decoded parameter value.
   */
  public String decodePathParam(String parameter) {
    parameter = urlDecode(parameter);
    for (String key : charReplacementMap.keySet()) {
      while (parameter.contains(charReplacementMap.get(key))) {
        parameter = parameter.replace(charReplacementMap.get(key), key);
      }

    }
    return parameter;
  }


  /**
   * Encodes url.
   *
   * @param url Url string to be encoded.
   * @return Encoded url.
   */
  public String urlEncode(String url) {
    try {
      String encoded = URLEncoder.encode(url, "UTF-8");
      return encoded;
    } catch (UnsupportedEncodingException e) {
      throw new InvocationException(e);
    }
  }

  /**
   * Decodes url.
   *
   * @param encoded Encoded url string.
   * @return Decoded url.
   */
  public String urlDecode(String encoded) {
    try {
      String decoded = URLDecoder.decode(encoded, "UTF-8");
      return decoded;
    } catch (UnsupportedEncodingException e) {
      throw new InvocationException(e);
    }
  }

}
