/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.web.common.client;

import org.appops.core.mime.MimeType;

/**
 * Wrapper base class which wraps actual web client instance e.g. Jetty HttpClient instance.
 *
 * @author deba
 * @version $Id: $Id
 */
public abstract class WebClient<W> {

  private W wrappedClient;

  /**
   * <p>Constructor for WebClient.</p>
   *
   * @param wrappedClient a W object.
   */
  protected WebClient(W wrappedClient) {
    setWrappedClient(wrappedClient);
  }

  /**
   * <p>Constructor for WebClient.</p>
   */
  protected WebClient() {

  }

  /**
   * Sends a get request through wrapped client instance.
   *
   * @param url Request url.
   * @return Response obtained from server.
   */
  public abstract ServerResponse get(String url);

  /**
   * Sends a get request through wrapped client instance.
   *
   * @param url Request url.
   * @param cookie Cookie to be sent with request.
   * @return Response obtained from server.
   */
  public abstract ServerResponse get(String url, Cookie cookie);

  /**
   * Sends a post request through wrapped client instance.
   *
   * @param content Content to be posted on server.
   * @param contentType Mimetype of content.
   * @param url Request url.
   * @return Response obtained from server.
   */
  public abstract ServerResponse post(String url, Object content, MimeType contentType);

  /**
   * Sends a post request through wrapped client instance.
   *
   * @param url Request url.
   * @param content Content to be posted on server.
   * @param contentType Mimetype of content.
   * @param cookie Cookie to be sent with request.
   * @return Response obtained from server.
   */
  public abstract ServerResponse post(String url, Object content, MimeType contentType,
      Cookie cookie);

  /**
   * Adds cookie to client.
   *
   * @param url Url for which cookie is to be added.
   * @param cookie to be added.
   */
  public abstract void addCookie(String url, Cookie cookie);

  /**
   * <p>addProxy.</p>
   *
   * @param proxy a {@link org.appops.web.common.client.Proxy} object.
   */
  public abstract void addProxy(Proxy proxy);

  /**
   * <p>wrappedClient.</p>
   *
   * @return a W object.
   */
  public W wrappedClient() {
    return wrappedClient;
  }

  /**
   * <p>Setter for the field <code>wrappedClient</code>.</p>
   *
   * @param wrappedClient a W object.
   */
  public void setWrappedClient(W wrappedClient) {
    this.wrappedClient = wrappedClient;
  }



}
