/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.slim.base.api;

import java.util.ArrayList;
import java.util.Collection;
import org.appops.core.service.Parameter;
import org.appops.core.service.RequestMethod;
import org.appops.core.service.ServiceRoute;
import org.appops.core.service.annotation.ServiceOp;
import org.appops.core.service.meta.InterfaceMeta;
import org.appops.core.service.meta.ServiceMeta;
import org.appops.core.service.meta.ServiceOpMeta;
import org.appops.slim.base.core.ServiceStore;

/**
 * <p>ServiceMetaManager interface.</p>
 *
 * @author deba
 * @version $Id: $Id
 */
@ServiceStore
public interface ServiceMetaManager {

  /**
   * Registers service meta.
   *
   * @param serviceMeta Service meta to be registered.
   */
  @ServiceOp(path = "register-meta", method = RequestMethod.POST)
  public void registerServiceMeta(ServiceMeta serviceMeta);


  /**
   * Provides service definition instance.
   *
   * @param name Name of service.
   * @return Instance that contains service meta information.
   */
  @ServiceOp(path = "service-meta/{name}", method = RequestMethod.GET)
  public ServiceMeta getServiceMeta(String name);

  /**
   * Provides service definition instance.
   *
   * @param name Name of service.
   * @return Instance that contains service meta information.
   */
  @ServiceOp(path = "service-meta/{name}/route", method = RequestMethod.GET)
  public ServiceRoute getServiceRoute(String name);

  /**
   * Provides service operation definition.
   *
   * @param path Service operation path or signature.
   * @return Instance that contains service operation meta information.
   */
  @ServiceOp(path = "op-meta-by-path/{path}", method = RequestMethod.GET)
  public ServiceOpMeta getOpMeta(String path);

  /**
   * Provides service operation definition.
   *
   * @param service Service name.
   * @param path Service operation path or signature.
   * @return Instance that contains service operation meta information.
   */
  @ServiceOp(path = "op-meta-by-path/{service}/{path}", method = RequestMethod.GET)
  public ServiceOpMeta getOpMeta(String service, String path);

  /**
   * Provides service operation definition.
   *
   * @param id Service operation id.
   * @return Instance that contains service operation meta information.
   */
  @ServiceOp(path = "op-meta-by-id/{id}", method = RequestMethod.GET)
  public ServiceOpMeta getOpMeta(Long id);

  /**
   * <p>readAllServices.</p>
   *
   * @return a {@link java.util.Collection} object.
   */
  @ServiceOp(path = "services/all", method = RequestMethod.GET)
  public Collection<ServiceMeta> readAllServices();

  /**
   * <p>getServicesByPage.</p>
   *
   * @param startIndex a {@link java.lang.Integer} object.
   * @param pageSize a {@link java.lang.Integer} object.
   * @return a {@link java.util.Collection} object.
   */
  @ServiceOp(path = "services/by-page", method = RequestMethod.GET)
  public Collection<ServiceMeta> getServicesByPage(Integer startIndex, Integer pageSize);

  /**
   * <p>getServiceInterfaces.</p>
   *
   * @param serviceName a {@link java.lang.String} object.
   * @return a {@link java.util.Collection} object.
   */
  @ServiceOp(path = "service/interfaces", method = RequestMethod.GET)
  public Collection<InterfaceMeta> getServiceInterfaces(String serviceName);

  /**
   * <p>getOperations.</p>
   *
   * @param serviceName a {@link java.lang.String} object.
   * @param interfaceName a {@link java.lang.String} object.
   * @return a {@link java.util.Collection} object.
   */
  @ServiceOp(path = "operations", method = RequestMethod.GET)
  public Collection<ServiceOpMeta> getOperations(String serviceName, String interfaceName);

  /**
   * <p>getFilteredServices.</p>
   *
   * @param filter a {@link java.lang.String} object.
   * @return a {@link java.util.Collection} object.
   */
  @ServiceOp(path = "getFilteredServices", method = RequestMethod.GET)
  public Collection<ServiceMeta> getFilteredServices(String filter);

  /**
   * <p>getOperationParameters.</p>
   *
   * @param serviceName a {@link java.lang.String} object.
   * @param interfaceName a {@link java.lang.String} object.
   * @param operationName a {@link java.lang.String} object.
   * @return a {@link java.util.ArrayList} object.
   */
  @ServiceOp(path = "getOperationParameters", method = RequestMethod.GET)
  ArrayList<Parameter> getOperationParameters(String serviceName, String interfaceName,
      String operationName);

}
