/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */
package org.appops.slim.base.invocation.async;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import org.appops.core.annotation.AsyncOf;
import org.appops.core.service.Parameter;
import org.appops.core.service.annotation.Service;
import org.appops.core.service.annotation.ServiceOp;
import org.appops.core.service.meta.InterfaceMeta;
import org.appops.core.service.meta.ServiceMeta;
import org.appops.core.service.meta.ServiceOpMeta;
import org.appops.slim.base.exception.InvocationException;

/**
 * A class which computes the Async operation Op meta by looking sync operation Op meta.
 *
 */
public class AsyncOpMetaComputer {

  /**
   * Creates sync service operation meta instance from method and arguments provided.
   * 
   * @param method method definition from which meta information is to be collected.
   * @param args Argument values passed to method while invocation.
   * @return computed {@link ServiceOpMeta}.
   * @throws NoSuchMethodException Thrown when particular method not found.
   * @throws SecurityException Thrown by the security manager to indicate security violation.
   */
  public ServiceOpMeta computeOpMeta(Method method, Object[] args)
      throws NoSuchMethodException, SecurityException {

    ServiceOpMeta opMeta = new ServiceOpMeta();
    opMeta.setName(method.getName());

    Class<?> asyncOfClass = method.getDeclaringClass();
    AsyncOf asyncOfAnnoration = asyncOfClass.getAnnotation(AsyncOf.class);

    Class syncClazz = asyncOfAnnoration.sync();
    Method syncMethod = syncClazz.getMethod(method.getName(), method.getParameterTypes());
    ServiceOp opAnnotation = syncMethod.getAnnotation(ServiceOp.class);

    if (opAnnotation == null) {
      throw new InvocationException(
          "Cannot invoke operation, ServiceOp annotation is absent on method ->"
              + syncMethod.getName());
    }

    opMeta.setFriendly(opAnnotation.friendly());
    opMeta.setPath(opAnnotation.path());
    opMeta.setResultTypeName(syncMethod.getReturnType().getCanonicalName());
    opMeta.setMethod(opAnnotation.method());
    opMeta.setAsyncFlag(opAnnotation.async());
    java.lang.reflect.Parameter[] parameters = syncMethod.getParameters();
    if (args != null) {
      for (int i = 0; i < args.length; i++) {
        Parameter parameter = new Parameter(i, parameters[i].getName(), args[i]);
        parameter.setTypeName(syncMethod.getParameterTypes()[i].getCanonicalName());
        opMeta.addParameter(parameter);
      }
    }
    InterfaceMeta interfaceMeta = new InterfaceMeta();
    interfaceMeta.setName(syncMethod.getDeclaringClass().getSimpleName());
    interfaceMeta.setQualifiedClassName(syncMethod.getDeclaringClass().getCanonicalName());
    opMeta.setParent(interfaceMeta);
    interfaceMeta.addOperation(opMeta);
    String serviceName = null;
    for (Annotation annotation : syncMethod.getDeclaringClass().getAnnotations()) {
      if (annotation.annotationType().isAnnotationPresent(Service.class)) {
        serviceName = annotation.annotationType().getSimpleName();
        break;
      }
    }
    if (serviceName == null) {
      throw new InvocationException(
          "Cannot invoke operation, unable to locate service name for method ->"
              + syncMethod.getName());
    }
    ServiceMeta serviceMeta = new ServiceMeta(serviceName);
    interfaceMeta.setParent(serviceMeta);
    return opMeta;

  }

}
