/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */
package org.appops.slim.base.invocation.async;

import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.name.Named;
import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import org.appops.core.constant.CoreGuiceConstant;
import org.appops.core.deployment.ServiceConfiguration;
import org.appops.core.mime.MimeType;
import org.appops.core.service.OpParameterMap;
import org.appops.core.service.Parameter;
import org.appops.core.service.RequestMethod;
import org.appops.core.service.ServiceRoute;
import org.appops.core.service.meta.ServiceOpMeta;
import org.appops.marshaller.DescriptorType;
import org.appops.marshaller.Marshaller;
import org.appops.slim.base.api.ServiceMetaManager;
import org.appops.slim.base.exception.InvocationException;
import org.appops.slim.base.invocation.UrlEncodeUtil;
import org.appops.web.common.client.AsyncWebClient;
import org.asynchttpclient.Response;

/**
 * Sends a rest request using http connection to service deployed in stand alone mode and converts
 * response data into appropriate type.
 * 
 */
public class AsyncApiRestInvoker<T> {

  private Provider<ServiceConfiguration> currentDeployment;
  private Provider<AsyncWebClient> asyncWebClient;
  private Marshaller marshaller;
  private Provider<ServiceMetaManager> serviceStore;

  /**
   * Sends rest request to service API and fetches result if expected.
   * 
   * @param opMeta Meta information of an operation which is to be invoked.
   * @return Result obtained from rest call.
   */
  public CompletableFuture<?> invokeRest(ServiceOpMeta opMeta) {
    try {
      String restUrl = getBaseRestUrl(opMeta);
      AsyncWebClient<?> webClient = getAsyncWebClient();
      CompletableFuture<Response> responseFuture = null;
      if (RequestMethod.GET.equals(opMeta.getMethod())) {
        restUrl = buildUrl(restUrl, opMeta, true);
        responseFuture = webClient.get(restUrl);
      } else if (RequestMethod.POST.equals(opMeta.getMethod())) {
        restUrl = buildUrl(restUrl, opMeta, false);
        String parameterJson = createParameterJson(opMeta.getParameters());
        responseFuture = webClient.post(restUrl, parameterJson, MimeType.JSON);
      } else {
        throw new InvocationException(
            "No request method provided in operation meta, please provide - GET/POST");
      }
      return responseFuture.thenApply(new Function<Response, Object>() {
        @Override
        public Object apply(Response t) {
          return convertResult(t, opMeta.getResultTypeName());
        }
      });
    } catch (Exception e) {
      throw ((e instanceof InvocationException) ? (InvocationException) e
          : new InvocationException(e));
    }
  }

  /**
   * It returns the base url using given service op meta.
   * 
   * @param opMeta Service Op meta
   * @return base url using given service op meta
   */
  private String getBaseRestUrl(ServiceOpMeta opMeta) {
    String deploymentUrl = getCurrentDeployment().getGatewayUrl();
    if (opMeta.serviceName().equals("ServiceStore")) {
      return getCurrentDeployment().getGatewayUrl();
    }
    ServiceRoute serviceRoute = getServiceStore().getServiceRoute(opMeta.serviceName());
    String serviceUrl = serviceRoute != null ? serviceRoute.getServiceUrl() : "";
    return (serviceUrl != null && !serviceUrl.isEmpty()) ? serviceUrl : deploymentUrl;
  }

  /**
   * It returns json of given parameters.
   * 
   * @param parameters instance of {@link OpParameterMap}
   * @return json of given op parameter map
   */
  private String createParameterJson(OpParameterMap parameters) {
    return marshaller.marshall(parameters, DescriptorType.JSON);
  }

  /**
   * Convert server response into appropriate type
   * 
   * @param response server response
   * @param resultTypename type in which data to be convert
   * @return actual object of result.
   */
  private Object convertResult(Response response, String resultTypename) {
    if (String.class.getCanonicalName().equals(resultTypename)) {
      return response.getResponseBody();
    }
    Class<?> resultType = null;
    try {
      resultType = Class.forName(resultTypename);
    } catch (Exception e) {
      throw new InvocationException("Error while converting rest call result", e);
    }

    return getMarshaller().unmarshall(response.getResponseBody(), resultType, DescriptorType.JSON);
  }

  /**
   * Builds complete rest url to be used to send rest request. The url also contains parameters to
   * be passed while sending request.
   * 
   * @param restUrl Service url.
   * @param opMeta Meta information of service api to be called.
   * @param appendParamters ppend parameters flag
   * @return Full rest url containing all information needed.
   */
  public String buildUrl(String restUrl, ServiceOpMeta opMeta, boolean appendParamters) {
    if (!restUrl.endsWith("/")) {
      restUrl += "/";
    }

    if (opMeta.serviceName() == null || opMeta.serviceName().isEmpty()) {
      throw new InvocationException("Parent service name is null, please provide it "
          + "through service op meta instance passed.");
    }
    restUrl += /* "Op/" + */ opMeta.serviceName();

    Collection<Parameter> params = opMeta.getParameters().values();
    String friendly = opMeta.getFriendly();
    String name = opMeta.getName();
    String path = opMeta.getPath();
    if (path != null && !path.isEmpty()) {
      return restUrl += ("/" + evaluatePathParameters(path, opMeta));
    } else {
      if (friendly != null && !friendly.isEmpty()) {
        restUrl += ("/" + friendly);
      } else if (name != null && !name.isEmpty()) {
        restUrl += ("/" + name);
      } else {
        throw new InvocationException(
            "Both signature and friendly cannot be null for an operation =>" + opMeta.getName()
                + ", Service => " + opMeta.serviceName());
      }
      if (appendParamters) {
        restUrl += addRequestParameters(params);
      }
      System.out.println("ApiRestInvoker url - > " + restUrl);
      return restUrl;
    }
  }

  /// **
  // * Converts jetty content response into appops response.
  // *
  // * @param contentResponse Jetty content response.
  // * @return Appops server response.
  // */
  // private ServerResponse convertToServerResponse(Response contentResponse) {
  // ServerResponse response = new ServerResponse();
  // response.setContent(contentResponse.getResponseBodyAsBytes());
  // response.setContentAsString(contentResponse.getResponseBody());
  // // response.setEncoding(contentResponse.);
  // response.setMediaType(contentResponse.getContentType());
  // response.setStatus(contentResponse.getStatusCode());
  // return response;
  // }
  private String evaluatePathParameters(String path, ServiceOpMeta opMeta) {
    UrlEncodeUtil parameterEncoder = new UrlEncodeUtil();
    if (path != null && path.contains("/")) {
      String[] pathElements = path.split("/");
      for (int i = 0; i < pathElements.length; i++) {
        String pathElement = pathElements[i];
        if (pathElement.startsWith("{") && pathElement.endsWith("}")) {
          String parameterPart =
              pathElement.substring(pathElement.indexOf("{"), pathElement.indexOf("}") + 1).trim();
          String parameterName =
              parameterPart.substring(parameterPart.indexOf("{") + 1, parameterPart.indexOf("}"));
          Parameter parameter = opMeta.getParameter(parameterName);
          if (parameter != null) {
            path = path.replace(parameterPart,
                parameterEncoder.encodePathParam(parameter.getValue().toString()));
          }
        }
      }
    }
    return path;
  }


  /**
   * Builds a string containing parameters to be appended to rest url.
   * 
   * @param params Parameter information e.g. name, value etc.
   * @return String containing parameter information which is to be appended to actual request url.
   */
  private String addRequestParameters(Collection<Parameter> params) {
    String parameterPart = "";

    if (!params.isEmpty()) {
      parameterPart += "?";
    }

    for (Parameter parameter : params) {
      if (!parameterPart.endsWith("?") && !parameterPart.endsWith("&")) {
        parameterPart += "&";
      }
      parameterPart += (parameter.getName() + "=" + parameter.getValue());
    }
    return parameterPart;
  }

  /**
   * <p>
   * Getter for the field <code>currentDeployment</code>.
   * </p>
   *
   * @return a {@link org.appops.core.deployment.DeploymentConfig} object.
   */
  public ServiceConfiguration getCurrentDeployment() {
    return currentDeployment.get();
  }

  /**
   * <p>
   * Setter for the field <code>currentDeployment</code>.
   * </p>
   *
   * @param currentDeployment a {@link com.google.inject.Provider} object.
   */
  @Inject
  public void setCurrentDeployment(
      @Named(CoreGuiceConstant.CURRENT_DEPLOYMENT) Provider<ServiceConfiguration> currentDeployment) {
    this.currentDeployment = currentDeployment;
  }

  public AsyncWebClient getAsyncWebClient() {
    return asyncWebClient.get();
  }

  @Inject
  public void setAsyncWebClient(Provider<AsyncWebClient> webClient) {
    this.asyncWebClient = webClient;
  }


  public Marshaller getMarshaller() {
    return marshaller;
  }


  @Inject
  public void setMarshaller(Marshaller marshaller) {
    this.marshaller = marshaller;
  }


  public ServiceMetaManager getServiceStore() {
    return serviceStore.get();
  }

  @Inject
  public void setServiceStore(Provider<ServiceMetaManager> serviceStore) {
    this.serviceStore = serviceStore;
  }
}
